/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.data;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.ametys.core.user.UserIdentity;

/**
 * Class representing a survey session, i.e. all the responses of a user to a survey.
 */
public class SurveySession
{
    
    /** The session ID. */
    protected int _id;
    
    /** The survey ID. */
    protected String _surveyId;
    
    /** The login of the person who answered the survey. */
    protected String _login;
    
    /** The population of the person who answered the survey. */
    protected String _population;
    
    /** The IP address of the person who answered the survey. */
    protected String _ipAddress;
    
    /** The date and time at which the person completed the survey. */
    protected Date _submittedAt;
    
    /** The list of answers. */
    protected List<SurveyAnswer> _answers;
    
    /**
     * Build a SurveySession object.
     */
    public SurveySession()
    {
        this(Integer.MIN_VALUE, null, null, null, null, new ArrayList<>());
    }
    
    /**
     * Build a SurveySession object.
     * @param id the session ID.
     * @param surveyId the survey ID.
     * @param user the user.
     * @param ipAddress the user IP address.
     * @param submittedAt the date of completion.
     * @param answers the list of answers.
     */
    public SurveySession(int id, String surveyId, UserIdentity user, String ipAddress, Date submittedAt, List<SurveyAnswer> answers)
    {
        this._id = id;
        this._surveyId = surveyId;
        this._login = user != null ? user.getLogin() : null;
        this._population = user != null ? user.getPopulationId() : null;
        this._ipAddress = ipAddress;
        this._submittedAt = submittedAt;
        this._answers = answers;
    }

    /**
     * Get the id.
     * @return the id
     */
    public int getId()
    {
        return _id;
    }

    /**
     * Set the id.
     * @param id the id to set
     */
    public void setId(int id)
    {
        this._id = id;
    }

    /**
     * Get the surveyId.
     * @return the surveyId
     */
    public String getSurveyId()
    {
        return _surveyId;
    }

    /**
     * Set the surveyId.
     * @param surveyId the surveyId to set
     */
    public void setSurveyId(String surveyId)
    {
        this._surveyId = surveyId;
    }

    /**
     * Get the login of the user answering the survey
     * @return the user login
     */
    public String getLogin()
    {
        return _login;
    }
    
    /**
     * Set the login of the user answering the survey
     * @param login the login
     */
    public void setLogin(String login)
    {
        _login = login;
    }
    
    /**
     * Get the population of the user answering the survey
     * @return the user identity
     */
    public String getPopulation()
    {
        return _population;
    }
    
    /**
     * Set the population of the user answering the survey
     * @param population the population
     */
    public void setPopulation(String population)
    {
        _population = population;
    }

    /**
     * Set the identity of the user answering the survey
     * @param user the user identity
     */
    public void setUser(UserIdentity user)
    {
        this._login = user != null ? user.getLogin() : "";
        this._population = user != null ? user.getPopulationId() : "";
    }
    
    /**
     * Get the identity of the user answering the survey
     * @return the user identity
     */
    public UserIdentity getUser()
    {
        if (this._login != null)
        {
            return new UserIdentity(this._login, this._population);
        }
        return null;
    }

    /**
     * Get the ipAddress.
     * @return the ipAddress
     */
    public String getIpAddress()
    {
        return _ipAddress;
    }

    /**
     * Set the ipAddress.
     * @param ipAddress the ipAddress to set
     */
    public void setIpAddress(String ipAddress)
    {
        this._ipAddress = ipAddress;
    }

    /**
     * Get the submittedAt.
     * @return the submittedAt
     */
    public Date getSubmittedAt()
    {
        return _submittedAt;
    }

    /**
     * Set the submittedAt.
     * @param submittedAt the submittedAt to set
     */
    public void setSubmittedAt(Date submittedAt)
    {
        this._submittedAt = submittedAt;
    }

    /**
     * Get the answers.
     * @return the answers
     */
    public List<? extends SurveyAnswer> getAnswers()
    {
        return _answers;
    }
    
    /**
     * Set the answers.
     * @param answers the answers to set
     */
    public void setAnswers(List<SurveyAnswer> answers)
    {
        this._answers = answers;
    }
    
}
