/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;

/**
 * {@link AmetysObject} for storing survey
 */
public class SurveyQuestion extends AbstractSurveyElement<SurveyQuestionFactory>
{
    /** Prefix for options */
    public static final String OPTION_NAME_PREFIX = "opt-";
    
    /** Constants for title metadata. */
    private static final String __PROPERTY_LABEL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":label";
    /** Constants for title metadata. */
    private static final String __PROPERTY_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";
    /** Constants for type metadata. */
    private static final String __PROPERTY_TYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":type";
    /** Constants for regexp metadata. */
    private static final String __PROPERTY_REGEXP = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":regexp";
    /** Constants for mandatory metadata. */
    private static final String __PROPERTY_MANDATORY = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":mandatory";
    /** Constants for other option metadata. */
    private static final String __PROPERTY_OTHER_OPTION = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":other-option";
    /** Constants for options metadata. */
    private static final String __NODE_NAME_OPTIONS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":options";
    /** Constants for columns metadata. */
    private static final String __NODE_NAME_COLUMNS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":columns";
    /** Constants for rules metadata. */
    private static final String __NODE_NAME_RULES = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rules";
    /** Constants for rule metadata. */
    private static final String __PROPERTY_RULE_TYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rule";
    private static final String __PROPERTY_RULE_OPTION = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":option";
    private static final String __PROPERTY_RULE_PAGE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":page";
    
    
    /** Type of a page. */
    public enum QuestionType
    {
        /** Free text. */
        FREE_TEXT,
        /** Multiline free text. */
        MULTILINE_FREE_TEXT,
        /** Single choice. */
        SINGLE_CHOICE,
        /** Multiple choice. */
        MULTIPLE_CHOICE,
        /** Matrix of single choice. */
        SINGLE_MATRIX,
        /** Matrix of multiple choice. */
        MULTIPLE_MATRIX
    }
    
    /**
     * Creates a {@link SurveyQuestion}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link SurveyFactory} which creates the AmetysObject.
     */
    public SurveyQuestion(Node node, String parentPath, SurveyQuestionFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Retrieves the label.
     * @return the label.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getLabel() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_LABEL).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get label property", e);
        }
    }
    
    /**
     * Set the label.
     * @param label the label.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setLabel(String label) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_LABEL, label);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set label property", e);
        }
    }

    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }
    
    /**
     * Retrieves the type.
     * @return the type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public QuestionType getType() throws AmetysRepositoryException
    {
        try
        {
            return QuestionType.valueOf(getNode().getProperty(__PROPERTY_TYPE).getString());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get type property", e);
        }
    }
    
    /**
     * Set the type.
     * @param type the type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setType(QuestionType type) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_TYPE, type.name());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set type property", e);
        }
    }
    
    /**
     * Retrieves the regexp type.
     * @return the regexp type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getRegExpType() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_REGEXP).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get regexp property", e);
        }
    }
    
    /**
     * Set the regexp type.
     * @param regexp the regexp type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setRegExpType(String regexp) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_REGEXP, regexp);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set regexp property", e);
        }
    }
    
    /**
     * Get the validation pattern.
     * @return the validation pattern.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getRegExpPattern () throws AmetysRepositoryException
    {
        String regexpType = getRegExpType();
        if (regexpType == null)
        {
            return null;
        }
        
        if ("int".equals(regexpType))
        {
            return "^-?[0-9]+$";
        }
        else if ("float".equals(regexpType))
        {
            return "^-?[0-9]+(\\.[0-9]+)?$";
        }
        else if ("email".equals(regexpType))
        {
            return "^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$";
        }
        else if ("phone".equals(regexpType))
        {
            return "^(\\+?\\(?[0-9]{1,3}\\)?([\\s]?)(\\(0\\))?|0)([\\s]?)([0-9\\-\\+\\s]{4,})+$";
        }
        else if ("date".equals(regexpType))
        {
            return "^([12][0-9][0-9][0-9])-([01][0-9])-([0123][0-9])$";
        }
        else if ("time".equals(regexpType))
        {
            return "^([012][0-9]):([012345][0-9])$";
        }
        else if ("datetime".equals(regexpType))
        {
            return "^([12][0-9][0-9][0-9])-([01][0-9])-([0123][0-9]) ([012][0-9]):([012345][0-9])$";
        }
        
        return null;
    }
    
    /**
     * Determines if the question is mandatory.
     * @return true if the question is mandatory.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean isMandatory() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_MANDATORY).getBoolean();
        }
        catch (PathNotFoundException e)
        {
            return false;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get mandatory property", e);
        }
    }
    
    /**
     * Set the mandatory.
     * @param mandatory true for mandatory
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setMandatory(boolean mandatory) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_MANDATORY, mandatory);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set mandatory property", e);
        }
    }
    
    /**
     * Determines if the question has a "other" option
     * @return true if the question has a "other" option
     */
    public boolean hasOtherOption ()
    {
        try
        {
            return getNode().getProperty(__PROPERTY_OTHER_OPTION).getBoolean();
        }
        catch (PathNotFoundException e)
        {
            return false;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get other option property", e);
        }
    }
    
    /**
     * Set the "other" option. To be used only for single or multiple choice question
     * @param other true to add the "other" option
     */
    public void setOtherOption (boolean other)
    {
        try
        {
            getNode().setProperty(__PROPERTY_OTHER_OPTION, other);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set other option property", e);
        }
    }
    
    /**
     * Set options
     * @param options the options as a Map of key, value
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setOptions (Map<String, String> options) throws AmetysRepositoryException
    {
        try
        {
            Node optsNode = getNode().getNode(__NODE_NAME_OPTIONS);
            NodeIterator itNode = optsNode.getNodes("ametys:*");

            // Remove old options
            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                node.remove();
            }
            
            for (String name : options.keySet())
            {
                Node optNode = optsNode.addNode("ametys:" + name, "ametys:survey-option");
                optNode.setProperty(__PROPERTY_TITLE, options.get(name));
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set options property", e);
        }
    }
    
    /**
     * Get options
     * @return the options as a Map of key, value
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Map<String, String> getOptions () throws AmetysRepositoryException
    {
        try
        {
            Map<String, String> options = new LinkedHashMap<>();
            
            NodeIterator itNode = getNode().getNode(__NODE_NAME_OPTIONS).getNodes("ametys:*");
            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                if (node.isNodeType("ametys:survey-option"))
                {
                    String value = node.getProperty(__PROPERTY_TITLE).getString();
                    options.put(node.getName().substring("ametys:".length()), value);
                }
            }
            
            return options;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get options property", e);
        }
    }
    
    /**
     * Set columns
     * @param columns the columns as a Map of key, value
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setColumns (Map<String, String> columns) throws AmetysRepositoryException
    {
        try
        {
            Node optsNode = getNode().getNode(__NODE_NAME_COLUMNS);
            NodeIterator itNode = optsNode.getNodes("ametys:*");

            // Remove old options
            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                node.remove();
            }
            
            for (String name : columns.keySet())
            {
                Node optNode = optsNode.addNode("ametys:" + name, "ametys:survey-option");
                optNode.setProperty(__PROPERTY_TITLE, columns.get(name));
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set columns property", e);
        }
    }
    
    /**
     * Get columns
     * @return the columns as a Map of key, value
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Map<String, String> getColumns () throws AmetysRepositoryException
    {
        try
        {
            Map<String, String> options = new LinkedHashMap<>();
            
            NodeIterator itNode = getNode().getNode(__NODE_NAME_COLUMNS).getNodes("ametys:*");
            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                if (node.isNodeType("ametys:survey-option"))
                {
                    String value = node.getProperty(__PROPERTY_TITLE).getString();
                    options.put(node.getName().substring("ametys:".length()), value);
                }
            }
            
            return options;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get columns property", e);
        }
    }
    
    /**
     * Add a rule for branching
     * @param option the chosen option
     * @param ruleType the rule type
     * @param page the page to jump or skip. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void addRules (String option, RuleType ruleType, String page) throws AmetysRepositoryException
    {
        try
        {
            if (!getNode().hasNode(__NODE_NAME_RULES))
            {
                getNode().addNode(__NODE_NAME_RULES, "ametys:survey-rules");
            }
            
            Node rulesNode = getNode().getNode(__NODE_NAME_RULES);
            
            Node ruleNode = rulesNode.addNode(option, "ametys:survey-rule");
            ruleNode.setProperty(__PROPERTY_RULE_TYPE, ruleType.name());
            ruleNode.setProperty(__PROPERTY_RULE_OPTION, option);
            if (ruleType == RuleType.JUMP || ruleType == RuleType.SKIP)
            {
                ruleNode.setProperty(__PROPERTY_RULE_PAGE, page);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add rule", e);
        }
    }
    
    /**
     * Determines if a rule with given option exists
     * @param option the option
     * @return true if teh rule exists
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasRule (String option) throws AmetysRepositoryException
    {
        try
        {
            Node rulesNode = getNode().getNode(__NODE_NAME_RULES);
            return rulesNode.hasNode(option);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to check if rule exists", e);
        }
    }
    
    /**
     * Delete a rule
     * @param option the option to delete
     * @throws AmetysRepositoryException  if an error occurs.
     */
    public void deleteRule (String option) throws AmetysRepositoryException
    {
        try
        {
            Node rulesNode = getNode().getNode(__NODE_NAME_RULES);
            
            if (rulesNode.hasNode(option))
            {
                rulesNode.getNode(option).remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to delete rule", e);
        }
    }
    
    /**
     * Get the rules
     * @return the rules
     * @throws AmetysRepositoryException if an error occurs.
     */
    public List<SurveyRule> getRules () throws AmetysRepositoryException
    {
        try
        {
            List<SurveyRule> rules = new LinkedList<>();
            
            NodeIterator itNode = getNode().getNode(__NODE_NAME_RULES).getNodes("*");
            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                if (node.isNodeType("ametys:survey-rule"))
                {
                    String option = node.getProperty(__PROPERTY_RULE_OPTION).getString();
                    RuleType type = RuleType.valueOf(node.getProperty(__PROPERTY_RULE_TYPE).getString());
                    String page = null;
                    if (node.hasProperty(__PROPERTY_RULE_PAGE))
                    {
                        page = node.getProperty(__PROPERTY_RULE_PAGE).getString();
                    }
                    rules.add(new SurveyRule(option, type, page));
                }
            }
            
            return rules;
        }
        catch (PathNotFoundException e)
        {
            return new LinkedList<>();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get rules property", e);
        }
    }
    
    /**
     * Get the page to which this question belongs.
     * @return the page to which this question belongs.
     * @throws AmetysRepositoryException if a repository error occurs when retrieving the page attached to a survey
     */
    public SurveyPage getSurveyPage() throws AmetysRepositoryException
    {
        return getParent();
    }
    
    /**
     * Get the survey to which this question belongs.
     * @return the survey to which this question belongs.
     * @throws AmetysRepositoryException if a repository error occurs when retrieving a survey
     */
    public Survey getSurvey() throws AmetysRepositoryException
    {
        return getSurveyPage().getSurvey();
    }
    
    @Override
    public SurveyQuestion copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        SurveyQuestion question = parent.createChild(name, "ametys:survey-question");
        question.setLabel(getLabel());
        question.setTitle(getTitle());
        question.setMandatory(isMandatory());
        question.setType(getType());
        
        String regExpType = getRegExpType();
        if (StringUtils.isNotEmpty(regExpType))
        {
            question.setRegExpType(regExpType);
        }
        
        if (getType().equals(QuestionType.MULTIPLE_CHOICE) || getType().equals(QuestionType.SINGLE_CHOICE))
        {
            question.setOtherOption(hasOtherOption());
        }
       
        question.setOptions(getOptions());
        question.setColumns(getColumns());
        
        for (SurveyRule rule : getRules())
        {
            question.addRules(rule.getOption(), rule.getType(), rule.getPage());
        }
        
        copyPictureTo(question);
        
        return question;
    }
    
    @Override
    public SurveyQuestion copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
}
