/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.syndication;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.concurrent.CountDownLatch;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.io.CloseMode;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.HttpUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import com.rometools.rome.feed.synd.SyndFeed;
import com.rometools.rome.io.FeedException;
import com.rometools.rome.io.SyndFeedInput;
import com.rometools.rome.io.XmlReader;

/**
 * Feed cache, supports preloading multiple feeds in multiple concurrent threads.
 */
public class FeedCache extends AbstractLogEnabled implements Component, Initializable, Serviceable, Disposable
{
    /** The component role. */
    public static final String ROLE = FeedCache.class.getName();
    
    /** The feed cache id */
    protected static final String CACHE_ID = ROLE + "$feedCache";
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The user information cache. */
    protected Cache<String, FeedResult> _cache;
    
    private CloseableHttpClient _httpClient;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        int timeoutValue = Config.getInstance().getValue("syndication.timeout", false, 2000L).intValue();
        _httpClient = HttpUtils.createHttpClient(-1, timeoutValue);
        
        _cacheManager.createMemoryCache(
                CACHE_ID,
                new I18nizableText("plugin.syndication", "PLUGINS_SYNDICATION_FEED_CACHE_LABEL"),
                new I18nizableText("plugin.syndication", "PLUGINS_SYNDICATION_FEED_CACHE_DESC"),
                true,
                Duration.ofDays(1));
        
        _cache = _cacheManager.get(CACHE_ID);
    }
    
    /**
     * Pre-load a collection of feeds, one by thread, to avoid getting timeouts sequentially.
     * @param feeds the feeds to preload.
     */
    public void preload(Collection<String> feeds)
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Preloading " + feeds.size() + " feeds...");
        }
        
        long start = System.currentTimeMillis();
        
        // Synchronization helper.
        CountDownLatch doneSignal = new CountDownLatch(feeds.size());
        
        for (String feed : feeds)
        {
            // Test if the feed is already in the cache.
            if (!_cache.hasKey(feed))
            {
                // The feed is not in the cache: prepare and launch a thread to retrieve it.
                new Thread(() -> {
                    try
                    {
                        if (getLogger().isDebugEnabled())
                        {
                            getLogger().debug("Preparing to load the URL '" + feed + "'");
                        }
                        
                        long tStart = System.currentTimeMillis();
                        
                        // Load the feed and signal that it's done.
                        _cache.get(feed, u -> FeedCache.loadFeed(u, _httpClient, getLogger()));
                        
                        if (getLogger().isDebugEnabled())
                        {
                            long tEnd = System.currentTimeMillis();
                            getLogger().debug("URL '" + feed + "' was loaded successfully in " + (tEnd - tStart) + " millis.");
                        }
                    }
                    catch (Exception e)
                    {
                        getLogger().error("An error occurred loading the URL '" + feed + "'", e);
                    }
                    finally
                    {
                        // Signal that the worker has finished.
                        doneSignal.countDown();
                    }
                }).start();
            }
            else
            {
                // The feed is already in the cache: count down.
                doneSignal.countDown();
            }
        }
        
        try
        {
            // Wait for all the URLs to be loaded (i.e. all the threads to end).
            doneSignal.await();
            
            if (getLogger().isDebugEnabled())
            {
                long end = System.currentTimeMillis();
                getLogger().debug("Feed preloading ended in " + (end - start) + " millis.");
            }
        }
        catch (InterruptedException e)
        {
            // Ignore, let the threads finish or die.
        }
    }
    
    /**
     * Get a feed not from the cache.
     * @param feedUrl the feed.
     * @return the feed result.
     * @throws IOException if an error occurs while loading the feed
     */
    public FeedResult getFeedNoCache(String feedUrl) throws IOException
    {
        return loadFeed(feedUrl, _httpClient, getLogger());
    }
    
    /**
     * Get a feed.
     * @param feedUrl the feed.
     * @param lifeTime the amount of date or time to be added to the field
     * @return the feed response.
     */
    public FeedResult getFeed(String feedUrl, int lifeTime)
    {
        FeedResult feedResult = _cache.get(feedUrl, u -> loadFeed(u, _httpClient, getLogger()));
        ZonedDateTime dateFeed = feedResult.getCreationDate();
        
        if (ZonedDateTime.now().isAfter(dateFeed.plusMinutes(lifeTime)))
        {
            _cache.invalidate(feedUrl);
            return _cache.get(feedUrl, u -> loadFeed(u, _httpClient, getLogger()));
        }
        else
        {
            return feedResult;
        }
    }
    
    /**
     * Retrieve a feed's content to store it into the cache.
     * @param feedUrl the feed to load.
     * @param httpClient the client to use to load the feed
     * @param logger the logger
     * @return the feed content.
     */
    protected static FeedResult loadFeed(String feedUrl, HttpClient httpClient, Logger logger)
    {
        FeedResult result = new FeedResult();
        
        result.setCreationDate(ZonedDateTime.now());
        
        try
        {
            HttpGet feedSource = new HttpGet(feedUrl);
            
            httpClient.execute(feedSource, response -> {
                int statusCode = response.getCode();
                if (statusCode != 200)
                {
                    result.setStatus(FeedResult.STATUS_ERROR);
                    result.setMessageError("Unable to join the RSS feed : " + feedUrl + ". HTTP response code is " + statusCode + ".");
                    logger.error("Unable to join the RSS feed : " + feedUrl + ". HTTP response code is " + statusCode + ".");
                    return result;
                }
                
                try (HttpEntity entity = response.getEntity();
                        InputStream is = entity.getContent();
                        Reader reader = new XmlReader(is))
                {
                    SyndFeedInput input = new SyndFeedInput();
                    SyndFeed synFeed = input.build(reader);
                    result.setStatus(FeedResult.STATUS_OK);
                    result.setResponse(synFeed);
                    
                    return result;
                }
                catch (IllegalArgumentException | FeedException e)
                {
                    throw new IOException("Unable to parse the feed due to previous exception", e);
                }
            });
            
        }
        catch (Exception e)
        {
            result.setStatus(FeedResult.STATUS_ERROR);
            result.setMessageError(e.getLocalizedMessage());
            logger.error("Unable to read the RSS feed to the url : " + feedUrl, e);
        }
        
        return result;
    }
    
    public void dispose()
    {
        _httpClient.close(CloseMode.GRACEFUL);
    }
}
