/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.syndication;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.HttpUtils;
import org.ametys.runtime.config.Config;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

import com.rometools.modules.mediarss.MediaEntryModule;
import com.rometools.modules.mediarss.MediaModule;
import com.rometools.modules.mediarss.types.MediaContent;
import com.rometools.modules.mediarss.types.MediaGroup;
import com.rometools.modules.mediarss.types.Metadata;
import com.rometools.modules.mediarss.types.Reference;
import com.rometools.modules.mediarss.types.Thumbnail;
import com.rometools.modules.mediarss.types.UrlReference;
import com.rometools.rome.feed.synd.SyndCategory;
import com.rometools.rome.feed.synd.SyndContent;
import com.rometools.rome.feed.synd.SyndEnclosure;
import com.rometools.rome.feed.synd.SyndEntry;
import com.rometools.rome.feed.synd.SyndFeed;
import com.rometools.rome.feed.synd.SyndImage;
import com.rometools.rome.feed.synd.SyndPerson;

/**
 * Generates the contents of an external RSS of Atom feed.<br>
 * This implementation is based on the ROME API.
 */
public class FeedGenerator extends ServiceableGenerator
{
    /** The feed cache */
    protected FeedCache _feedCache;
    
    /** The rendering context handler. */
    protected RenderingContextHandler _renderingContext;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _feedCache = (FeedCache) serviceManager.lookup(FeedCache.ROLE);
        _renderingContext = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        boolean isCustom = false;
        boolean isSelected = false;
        long length = -1;
        String url;
        String name = "";
        int lifeTime = 0;
        
        @SuppressWarnings("unchecked")
        Map<String, Object> ctxParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        if (ctxParameters != null)
        {
            url = (String) ctxParameters.get("url");
            name = (String) ctxParameters.get("name");
            lifeTime = (Integer) ctxParameters.get("cache");
            length = (Long) ctxParameters.get("length");
            isCustom = (Boolean) ctxParameters.get("isCustom");
            isSelected = (Boolean) ctxParameters.get("isSelected");
        }
        else
        {
            url = source;
        }
        
        contentHandler.startDocument();

        if (_checkForInfiniteLoop(url))
        {
            getLogger().error("Infinite loop detected for RSS feed : " + url + ". The RSS feed url can not be the page itself.");
            if (isCustom && _renderingContext.getRenderingContext() == RenderingContext.FRONT)
            {
                _saxErrorFeed("feed-error-custom", url, name, null);
            }
            else if (_renderingContext.getRenderingContext() == RenderingContext.BACK)
            {
                _saxErrorFeed("feed-error", url, name, "Infinite loop detected for RSS feed : " + url + ". The RSS feed url can not be the page itself.");
            }
        }
        else
        {
            FeedResult feedResult = _feedCache.getFeed(url, lifeTime);
            
            if (feedResult.getStatus() == FeedResult.STATUS_OK)
            {
                SyndFeed feed = feedResult.getSynFeed();
                _saxFeeds(feed, length, url, isCustom, isSelected);
            }
            else if (_renderingContext.getRenderingContext() == RenderingContext.BACK)
            {
                _saxErrorFeed("feed-error", url, name, feedResult.getMessageError());
            }
            else if (isCustom && _renderingContext.getRenderingContext() == RenderingContext.FRONT)
            {
                _saxErrorFeed("feed-error-custom", url, name, null);
            }
        }

        contentHandler.endDocument();
    }
    
    private void _saxErrorFeed (String errorName, String url, String name, String messageError) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        _addAttribute(atts, "url", url);
        _addAttribute(atts, "name", name);
        if (StringUtils.isNotEmpty(messageError))
        {
            _addAttribute(atts, "messageError", messageError);
        }

        XMLUtils.createElement(contentHandler, errorName, atts);
    }
    
    private void _saxFeeds (SyndFeed feed, long length, String url, Boolean isCustom, Boolean isSelected) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();

        _addAttribute(atts, "title", feed.getTitle());
        _addAttribute(atts, "description", feed.getDescription());
        _addAttribute(atts, "feedUrl", url);
        _addAttribute(atts, "isCustom", String.valueOf(isCustom));
        _addAttribute(atts, "isSelected", String.valueOf(isSelected));

        XMLUtils.startElement(contentHandler, "feed", atts);
        
        SyndImage image = feed.getImage();
        
        if (image != null)
        {
            atts = new AttributesImpl();
            _addAttribute(atts, "link", image.getLink());
            _addAttribute(atts, "url", image.getUrl());
            _addAttribute(atts, "title", image.getTitle());
            
            XMLUtils.createElement(contentHandler, "image", atts);
        }
        
        List<SyndCategory> feedCategories = feed.getCategories();
        _saxCategories(feedCategories);
        
        List<SyndEntry> entries = feed.getEntries();
        if (entries != null)
        {
            int index = 0;
            Iterator<SyndEntry> it = entries.iterator();
            
            while (it.hasNext() && (length == -1 || index < length))
            {
                SyndEntry entry = it.next();
                
                atts = new AttributesImpl();
                _addAttribute(atts, "title", entry.getTitle());
                _addAttribute(atts, "link", entry.getLink());
    
                Date date = entry.getPublishedDate();
                if (date != null)
                {
                    _addAttribute(atts, "date", new SimpleDateFormat("yyyy-MM-dd'T'HH:mm").format(date));
                }
    
                XMLUtils.startElement(contentHandler, "entry", atts);
                
                SyndContent desc = entry.getDescription();
                if (desc != null && desc.getValue() != null)
                {
                    atts = new AttributesImpl();
                    _addAttribute(atts, "type", desc.getType());
                    XMLUtils.createElement(contentHandler, "description", atts, desc.getValue());
                }
                
                List<SyndPerson> authors = entry.getAuthors();
                if (authors != null)
                {
                    for (SyndPerson author : authors)
                    {
                        atts = new AttributesImpl();
                        _addAttribute(atts, "email", author.getEmail());
                        XMLUtils.createElement(contentHandler, "author", atts, author.getName());
                    }
                }
                
                // Attachments
                _saxEnclosures (entry);
                
                List<SyndCategory> entryCategories = entry.getCategories();
                _saxCategories(entryCategories);
                
                _saxMediaRSS(entry);
                
                XMLUtils.endElement(contentHandler, "entry");
                
                index++;
            }
        }
        
        XMLUtils.endElement(contentHandler, "feed");
    }

    private void _saxCategories(List<SyndCategory> feedCategories) throws SAXException
    {
        for (SyndCategory category : feedCategories)
        {
            AttributesImpl categoryAtts = new AttributesImpl();
            String taxonomyUri = category.getTaxonomyUri();
            if (taxonomyUri != null)
            {
                _addAttribute(categoryAtts, "domain", taxonomyUri);
            }
            XMLUtils.createElement(contentHandler, "category", categoryAtts, category.getName());
        }
    }
    
    private void _saxEnclosures (SyndEntry entry) throws SAXException
    {
        List<SyndEnclosure> enclosures = entry.getEnclosures();
        if (enclosures != null)
        {
            for (SyndEnclosure enclosure : enclosures)
            {
                AttributesImpl atts = new AttributesImpl();
                _addAttribute(atts, "type", enclosure.getType());
                _addAttribute(atts, "url", enclosure.getUrl());
                _addAttribute(atts, "name", _enclosureName(enclosure.getUrl()));
                XMLUtils.startElement(contentHandler, "enclosure", atts);
                _saxLength(enclosure.getLength());
                XMLUtils.endElement(contentHandler, "enclosure");
            }
        }
    }
    
    private void _saxMediaRSS(SyndEntry entry) throws SAXException
    {
        MediaEntryModule module = (MediaEntryModule) entry.getModule(MediaModule.URI);
        if (module != null)
        {
            XMLUtils.startElement(contentHandler, "media");
            
            for (MediaContent content : module.getMediaContents())
            {
                _saxMediaContent(content, false);
            }
            
            for (MediaGroup group : module.getMediaGroups())
            {
                AttributesImpl atts = new AttributesImpl();
                _addAttribute(atts, "defaultContentIndex", group.getDefaultContentIndex());
                XMLUtils.startElement(contentHandler, "mediagroup", atts);
                
                for (MediaContent content : group.getContents())
                {
                    _saxMediaContent(content, true);
                }
                
                XMLUtils.endElement(contentHandler, "mediagroup");
            }
            
            Metadata metadata = module.getMetadata();
            
            if (metadata != null)
            {
                saxMediaMetadata(metadata);
            }
            
            XMLUtils.endElement(contentHandler, "media");
        }
    }
    
    private void _saxMediaContent(MediaContent content, boolean inGroup) throws SAXException
    {
        Metadata metadata = content.getMetadata();
        Reference reference = content.getReference();
        
        AttributesImpl atts = new AttributesImpl();
        
        if (reference instanceof UrlReference)
        {
            _addAttribute(atts, "url", ((UrlReference) reference).getUrl().toString());
        }
        if (inGroup && content.isDefaultContent())
        {
            atts.addCDATAAttribute("default", "true");
        }
        _addAttribute(atts, "type", content.getType());
        _addAttribute(atts, "medium", content.getMedium());
        _addAttribute(atts, "size", content.getFileSize());
        _addAttribute(atts, "width", content.getWidth());
        _addAttribute(atts, "height", content.getHeight());
        
        XMLUtils.startElement(contentHandler, "mediacontent", atts);
        
        if (metadata != null)
        {
            saxMediaMetadata(metadata);
        }
        
        XMLUtils.endElement(contentHandler, "mediacontent");
    }
    
    private void saxMediaMetadata(Metadata metadata) throws SAXException
    {
        if (metadata != null)
        {
            String title = metadata.getTitle();
            Thumbnail[] thumbnails = metadata.getThumbnail();
            
            if (title != null)
            {
                AttributesImpl titleAttrs = new AttributesImpl();
                _addAttribute(titleAttrs, "type", metadata.getTitleType());
                XMLUtils.createElement(contentHandler, "title", titleAttrs, title);
            }
            
            for (Thumbnail thumbnail : thumbnails)
            {
                AttributesImpl thumbAttrs = new AttributesImpl();
                _addAttribute(thumbAttrs, "url", thumbnail.getUrl().toString());
                _addAttribute(thumbAttrs, "width", thumbnail.getWidth());
                _addAttribute(thumbAttrs, "height", thumbnail.getHeight());
                XMLUtils.createElement(contentHandler, "thumbnail", thumbAttrs);
            }
        }
    }
    
    private boolean _checkForInfiniteLoop (String src)
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Page page = (Page) request.getAttribute(Page.class.getName());
        
        if (page == null)
        {
            return false;
        }
        
        String cmsUrl = HttpUtils.sanitize(Config.getInstance().getValue("cms.url"));
        Site site = page.getSite();
        String siteName = page.getSiteName();
        String lang = page.getSitemapName();
        String path = page.getPathInSitemap();
        
        String frontUrl = site.getUrl() +  "/" + lang + "/" + path + ".html";
        if (src.equals(frontUrl))
        {
            return true;
        }
        
        String previewUrl = cmsUrl + "/preview/" + siteName + "/" + lang + "/" + path + ".html";
        if (src.equals(previewUrl))
        {
            return true;
        }
        
        String liveUrl = cmsUrl + "/live/" + siteName + "/" + lang + "/" + path + ".html";
        if (src.equals(liveUrl))
        {
            return true;
        }
        
        String backUrl = cmsUrl + siteName + "/" + lang + "/" + path + ".html";
        if (src.equals(backUrl))
        {
            return true;
        }
        
        return false;
    }
    
    private String _enclosureName (String url)
    {
        String name = url;
        if (name != null && name.lastIndexOf("/") > 0)
        {
            name = name.substring(name.lastIndexOf("/") + 1);
        }
        
        if (name != null && name.indexOf("?") > 0)
        {
            name = name.substring(0, name.indexOf("?"));
        }
        
        return name;
    }
    
    private void _addAttribute(AttributesImpl atts, String name, Object value)
    {
        if (value != null)
        {
            if (value instanceof String)
            {
                atts.addCDATAAttribute(name, (String) value);
            }
            else if (value instanceof Integer)
            {
                atts.addCDATAAttribute(name, ((Integer) value).toString());
            }
            else if (value instanceof Long)
            {
                atts.addCDATAAttribute(name, ((Long) value).toString());
            }
            else if (value instanceof Double)
            {
                atts.addCDATAAttribute(name, ((Double) value).toString());
            }
            else if (value instanceof Float)
            {
                atts.addCDATAAttribute(name, ((Float) value).toString());
            }
        }
    }
    
    private void _saxLength(long length) throws SAXException
    {
        org.ametys.core.util.StringUtils.toReadableDataSize(length).toSAX(contentHandler, "length");
    }
}
