/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.syndication;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Generator for feeds service
 */
public class FeedsGenerator extends ServiceableGenerator
{
    /** User pref key for urls */
    public static final String USER_PREF_RSS_URL_KEY = "user-pref-rss-urls";
    
    /** User pref key for positions */
    public static final String USER_PREF_RSS_ID_KEY = "user-pref-rss-positions";
    
    private static HashMap<String, Integer> _lifeTimes = new HashMap<>();
    static 
    {
        _lifeTimes.put("1", 1440); // 24 hours
        _lifeTimes.put("2", 180); // 3 hours
        _lifeTimes.put("3", 30); // 30 minutes
    }
    
    /** Conf access component */
    protected RssFeedUserPrefsComponent _confAccess;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The group manager */
    protected GroupManager _groupManager;
    
    /** The feed cache */
    protected FeedCache _feedCache;
    
    /** The source resolver */
    protected SourceResolver _resolver;
    
    /** The user preferences manager. */
    protected UserPreferencesManager _userPrefManager;
    
    /** The rendering context handler. */
    protected RenderingContextHandler _renderingContext;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    private JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _feedCache = (FeedCache) serviceManager.lookup(FeedCache.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _groupManager = (GroupManager) serviceManager.lookup(GroupManager.ROLE);
        _userPrefManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE + ".FO");
        _renderingContext = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
        _confAccess = (RssFeedUserPrefsComponent) serviceManager.lookup(RssFeedUserPrefsComponent.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        UserIdentity user = _currentUserProvider.getUser();
        
        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        String siteName = sitemapElement.getSiteName();
        String lang = sitemapElement.getSitemapName();
        
        String storageContext = siteName + "/" + lang + "/" + zoneItem.getId();
        Map<String, String> contextVars = _getContextVars(siteName, lang);
        
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();

        List<String> urlList = new ArrayList<>();
        List<String> idList = new ArrayList<>();
        try
        {
            // Check user prefence only if the page has limited access
            if (!_rightManager.hasAnonymousReadAccess(sitemapElement))
            {
                urlList = _getListUrl(user, storageContext, contextVars);
                idList = _getListId(user, storageContext, contextVars);
            }
        }
        catch (UserPreferencesException e1)
        {
            getLogger().error("Can't have the user preference from user : '" + user + "'");
        }
        
        contentHandler.startDocument();
        _saxAllFeed(request, zoneItem, serviceParameters, urlList, idList);
        contentHandler.endDocument();
    }
    
    // Sax all the feed depend on the service parameters
    private void _saxAllFeed(Request request, ZoneItem zoneItem, ModelAwareDataHolder serviceParameters, List<String> urlList, List<String> userSelectedFeedsId) throws SAXException
    {
        long defaultLength = serviceParameters.getValue("length", false, -1L);
        String defaultLifeTime = serviceParameters.getValue("cache");

        AttributesImpl atts = new AttributesImpl();

        ModelAwareRepeater rssFeeds = serviceParameters.getRepeater("feeds");
        List<ModelAwareRepeaterEntry> listFeeds = rssFeeds.getEntries().stream()
                                                                       .filter(entry -> _checkUserAccessRSS(entry))
                                                                       .collect(Collectors.toList());

        int nbMaxFeedFullSaxed = Math.toIntExact(serviceParameters.getValue("nb-feed-max", false, 0L));
        int nbMaxUser = Math.toIntExact(serviceParameters.getValue("nb-feed-user", false, 0L));
        
        atts.addCDATAAttribute("nbFeedService", String.valueOf(listFeeds.size()));
        atts.addCDATAAttribute("zoneItemId", String.valueOf(zoneItem.getId()));
        atts.addCDATAAttribute("nbMaxUser", String.valueOf(nbMaxUser));
        atts.addCDATAAttribute("nbMax", String.valueOf(nbMaxFeedFullSaxed));
        
        Page page = (Page) request.getAttribute(Page.class.getName());
        atts.addCDATAAttribute("showForm", String.valueOf(_confAccess.showPreferenceForm(page, nbMaxUser, listFeeds.size(), nbMaxFeedFullSaxed)));
        
        XMLUtils.startElement(contentHandler, "rssFeeds", atts);
        
        _saxFeeds(nbMaxUser, listFeeds, urlList, nbMaxFeedFullSaxed, defaultLength, defaultLifeTime, serviceParameters, userSelectedFeedsId);
        _saxFeedsConfig(nbMaxUser, listFeeds, urlList, userSelectedFeedsId);
        
        XMLUtils.endElement(contentHandler, "rssFeeds");
    }
    
    private void _saxFeedsConfig(long nbMaxUser, List<ModelAwareRepeaterEntry> listFeeds, List<String> urlList, List<String> userSelectedFeedsId) throws SAXException
    {
        for (ModelAwareRepeaterEntry rssFeed : listFeeds)
        {
            _saxInfoFeed(rssFeed.getValue("url"), rssFeed.getValue("title"), rssFeed.getValue("id"), false, userSelectedFeedsId.contains(rssFeed.getValue("id")));
        }
        
        for (int i = 1; i <= nbMaxUser; i++)
        {
            boolean isSelected = userSelectedFeedsId.contains("feed-url" + i) || userSelectedFeedsId.contains(String.valueOf(i + listFeeds.size()));
            if (i <= urlList.size())
            {
                String url  = urlList.get(i - 1);
                _saxInfoFeed(url, "", "feed-url" + i, true, isSelected);
            }
            else
            {
                _saxInfoFeed("", "", "feed-url" + i, true, isSelected);
            }
        }
    }
    
    private void _saxFeeds(int nbMaxUser, List<ModelAwareRepeaterEntry> listFeeds, List<String> urlList, long nbMaxFeedFullSaxed, long defaultLength, String defaultLifeTime, ModelAwareDataHolder serviceParameters,  List<String> userSelectedFeedsId)
    {
        int nbFeedFullSaxed = 0;
        for (ModelAwareRepeaterEntry rssFeed : listFeeds)
        {
            boolean isSelected = userSelectedFeedsId.contains(rssFeed.getValue("id"));
            if ((nbFeedFullSaxed < nbMaxFeedFullSaxed || nbMaxFeedFullSaxed == 0)
                    && (userSelectedFeedsId.isEmpty() || isSelected))
            {
                nbFeedFullSaxed++;
                _saxFullInfoFeed(rssFeed, isSelected, defaultLength, defaultLifeTime); // Sax all informations
            }
        }
        
        for (int i = 1; i <= nbMaxUser; i++)
        {
            if (i <= urlList.size())
            {
                String url = urlList.get(i - 1);
                boolean isSelected = userSelectedFeedsId.contains("feed-url" + i) || userSelectedFeedsId.contains(String.valueOf(i + listFeeds.size()));
                if ((nbFeedFullSaxed < nbMaxFeedFullSaxed || nbMaxFeedFullSaxed == 0)
                        && (userSelectedFeedsId.isEmpty() || isSelected))
                {
                    nbFeedFullSaxed++;
                    _saxFullInfoFeedCustom(serviceParameters, url, defaultLength, isSelected); // Sax all informations
                }
            }
        }
    }
    
    // Sax few info from the feed for the user preference form
    private void _saxInfoFeed(String url, String name, String feedId, boolean isCustom, boolean isSelected) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();

        atts.addCDATAAttribute("feedUrl", url);
        atts.addCDATAAttribute("feedName", name);
        atts.addCDATAAttribute("feedId", feedId);
        atts.addCDATAAttribute("isCustom", String.valueOf(isCustom));
        atts.addCDATAAttribute("isSelected", String.valueOf(isSelected));

        XMLUtils.createElement(contentHandler, "feed-conf", atts);
    }
    
    // Sax all info from the feed for the service (Feed from service parameters)
    private void _saxFullInfoFeed(ModelAwareRepeaterEntry rssFeed, boolean isSelected, long defaultLength, String defaultLifeTime)
    {
        long nbItems = rssFeed.getValue("length", false, Long.valueOf(defaultLength));
        String lifeTime = rssFeed.getValue("cache");
        if (StringUtils.isBlank(lifeTime))
        {
            lifeTime = defaultLifeTime;
        }
        
        String url = rssFeed.getValue("url");
        String name = rssFeed.getValue("title");
        
        _saxFeed(nbItems, url, name, lifeTime, false, isSelected);
    }
    
    // Sax all info from the feed for the service (Feed from user preference)
    private void _saxFullInfoFeedCustom(ModelAwareDataHolder serviceParameters, String url, long defaultLength, boolean isSelected)
    {
        String lifeTime = serviceParameters.getValue("cache");
        _saxFeed(defaultLength, url, "", lifeTime, true, isSelected);
    }
    
    // Call the pipeline to sax all information from a feed
    private void _saxFeed(long length, String url, String name, String lifeTime, Boolean isCustom, Boolean isSelected)
    {
        SitemapSource sitemapSource = null;
        try
        {
            HashMap<String, Object> parameter = new HashMap<>();
            parameter.put("length", length);
            parameter.put("url", url);
            parameter.put("name", name);
            parameter.put("cache", _lifeTimes.get(lifeTime));
            parameter.put("isCustom", isCustom);
            parameter.put("isSelected", isSelected);
            
            sitemapSource = (SitemapSource) _resolver.resolveURI("cocoon:/feed", null, parameter);
            sitemapSource.toSAX(new IgnoreRootHandler(contentHandler));
        }
        catch (Exception e)
        {
            getLogger().error("There is an error in the flux : '" + url + "'");
        }
        finally
        {
            _resolver.release(sitemapSource);
        }
    }
    
    // Get the list of url from the user preferences
    private List<String> _getListUrl(UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        if (user != null)
        {
            String urls = _userPrefManager.getUserPreferenceAsString(user, storageContext, contextVars, USER_PREF_RSS_URL_KEY);
            
            List<String> urlAsList = new ArrayList<>();
            if (StringUtils.isNotEmpty(urls))
            {
                String[] urlsAsArray = StringUtils.split(urls, ",");
                for (String url : urlsAsArray)
                {
                    urlAsList.add(URIUtils.decode(url));
                }
            }
              
            return urlAsList;
        }
        
        return new ArrayList<>();
        
    }
    
    // Get the list of chosen feed (unique id) from the user preferences
    private List<String> _getListId(UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        if (user != null)
        {
            String ids = _userPrefManager.getUserPreferenceAsString(user, storageContext, contextVars, USER_PREF_RSS_ID_KEY);
            if (StringUtils.isNotEmpty(ids))
            {
                String[] positionsAsArray = ids.split(",");
                return Arrays.asList(positionsAsArray);
            }
        }
        
        return new ArrayList<>();
    }
    
    // Check if the user can access to the feed
    private boolean _checkUserAccessRSS(ModelAwareRepeaterEntry rssFeed)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        Boolean limited = rssFeed.getValue("limited");
        if (limited)
        {
            // Return false if no user is logged
            return user != null;
        }
        else 
        {
            String[] foGroupsStr = rssFeed.getValue("fo-group", false, ArrayUtils.EMPTY_STRING_ARRAY);
            Set<GroupIdentity> foGroups = Arrays.asList(foGroupsStr).stream()
                    .filter(identityAsStr -> identityAsStr.length() > 0)
                    .map(identityAsStr -> _jsonUtils.convertJsonToMap(identityAsStr))
                    .map(identityAsMap -> new GroupIdentity((String) identityAsMap.get("groupId"), (String) identityAsMap.get("groupDirectory")))
                    .collect(Collectors.toSet());
            
            UserIdentity[] foUsersArray = rssFeed.getValue("fo-user", false, new UserIdentity[0]);
            Set<UserIdentity> foUsers = Arrays.asList(foUsersArray)
                                              .stream()
                                              .collect(Collectors.toSet());
            
            if (CollectionUtils.isEmpty(foUsers) && CollectionUtils.isEmpty(foGroups))
            {
                return true;
            }
            
            if (user == null)
            {
                return false;
            }
            
            if (foUsers.contains(user))
            {
                return true;
            }
            
            for (GroupIdentity foGroup : foGroups)
            {
                if (foGroup != null)
                {
                    Group group = _groupManager.getGroup(foGroup);
                    if (group != null && group.getUsers().contains(user))
                    {
                        return true;
                    }
                }
            }
            
            return false;
        }
    }
    
    private Map<String, String> _getContextVars(String siteName, String language)
    {
        Map<String, String> contextVars = new HashMap<>();
        
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_LANGUAGE, language);
        
        return contextVars;
    }
}


