/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.syndication;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;


/**
 *  Component for access to the user preferences for feeds
 */
public class RssFeedUserPrefsComponent implements Component, Serviceable
{
    /** Avalon ROLE. */
    public static final String ROLE = RssFeedUserPrefsComponent.class.getName();
 
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The rendering context handler. */
    protected RenderingContextHandler _renderingContext;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _renderingContext = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    /**
     * Determines if the preference forms for RSS Feeds must be displayed
     * @param page The page
     * @param nbCustomFeeds The number of personal feed
     * @param nbFeeds The number of feeds
     * @param maxFeeds The max number of feeds
     * @return true if the preference forms for RSS Feeds must be displayed
     */
    public boolean showPreferenceForm(Page page, int nbCustomFeeds, long nbFeeds, long maxFeeds)
    {
        if (page == null)
        {
            return false;
        }
        
        return !_rightManager.hasAnonymousReadAccess(page) && _hasCustomsFeeds(nbCustomFeeds, nbFeeds, maxFeeds);
    }

    /**
     * Determines if the service for feeds is cacheable
     * @param repeater The repeater feeds
     * @param nbCustomFeeds The number of personal feed
     * @param nbFeeds The number of feeds
     * @param maxFeeds The max number of feeds
     * @return true if the service is cacheable
     */
    public boolean isServiceCacheable(ModelAwareRepeater repeater, int nbCustomFeeds, int nbFeeds, int maxFeeds)
    {
        for (ModelAwareRepeaterEntry feed : repeater.getEntries())
        {
            boolean limited = feed.getValue("limited");
            UserIdentity[] foUsers = feed.getValue("fo-user", false, new UserIdentity[0]);
            String[] foGroups = feed.getValue("fo-group", false, ArrayUtils.EMPTY_STRING_ARRAY);
            
            if (limited || ArrayUtils.isNotEmpty(foUsers) || ArrayUtils.isNotEmpty(foGroups))
            {
                return false;
            }
        }
        
        return !_hasCustomsFeeds(nbCustomFeeds, nbFeeds, maxFeeds);
    }
    
    private boolean _hasCustomsFeeds(int nbCustomFeeds, long nbFeeds, long maxFeeds)
    {
        if (nbCustomFeeds != 0)
        {
            return true;
        }
        
        if (maxFeeds != 0 && maxFeeds < nbFeeds)
        {
            return true;
        }
        
        return false;
    }
}

