/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.syndication;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.userpref.FOUserPreferencesConstants;


/**
 * Set user preference
 */
public class SetRssFeedUserPreferencesAction extends ServiceableAction
{
    /** The user preferences manager. */
    protected UserPreferencesManager _userPrefManager;
    
    /** The Ametys resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userPrefManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE + ".FO");
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> result = new HashMap<>();
        List<I18nizableText> errors = new ArrayList<>();

        UserIdentity user = _currentUserProvider.getUser();
        
        String zoneItemId = request.getParameter("zone-item-id");
        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
   
        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        String language = sitemapElement.getSitemapName();
        String siteName = sitemapElement.getSiteName();
        
        String storageContext = siteName + "/" + language + "/" + zoneItem.getId();
        Map<String, String> contextVars = _getContextVars(siteName, language);
        
        Map<String, String> values = new HashMap<>();
        String ids = _getIds(request);
        
        int nbMaxCheck = Integer.parseInt(request.getParameter("nbMaxCheck"));
        if (StringUtils.split(ids, ",").length <= nbMaxCheck || nbMaxCheck == 0)
        {
            values.put(FeedsGenerator.USER_PREF_RSS_ID_KEY, ids);
            values.put(FeedsGenerator.USER_PREF_RSS_URL_KEY, _getUrls(request));
            
            _userPrefManager.setUserPreferences(user, storageContext, contextVars, values);
        }
        else
        {
            errors.add(new I18nizableText("plugin.syndication", "SYNDICATION_SERVICE_CONFIG_RSS_ERROR_TO_MANY_CHECK"));
        }
        
        if (errors.size() > 0)
        {
            result.put("error", errors);
        }
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        
        return EMPTY_MAP;
    }
    
    private String _getUrls(Request request)
    {
        int nbFeedUser = Integer.parseInt(request.getParameter("nbMaxUser"));

        int nbFeed = Integer.parseInt(request.getParameter("nbFeedService"));
        int position = nbFeed + 1;
        
        List<String> urls = new ArrayList<>();
        for (int i = 0; i < nbFeedUser; i++)
        {
            String url = request.getParameter("feed-url-" + (position + i));
            if (!StringUtils.isWhitespace(url))
            {
                urls.add(URIUtils.encodePath(url));
            }
        }
        
        return StringUtils.join(urls, ",");
    }
    
    private String _getIds(Request request)
    {
        int nbFeed = Integer.parseInt(request.getParameter("nbFeedService"));
        int nbFeedUser = Integer.parseInt(request.getParameter("nbMaxUser"));

        List<String> ids = new ArrayList<>();
        for (int i = 1; i <= (nbFeed + nbFeedUser); i++)
        {
            String checked = request.getParameter("feed-check-" + i);
            if (!StringUtils.isEmpty(checked))
            {
                if (checked.equals("true") || checked.equals("on"))
                {
                    String url = request.getParameter("feed-url-" + i);
                    if (!StringUtils.isWhitespace(url))
                    {
                        ids.add(request.getParameter("feed-id-" + i));
                    }
                }
            }
        }
        
        return StringUtils.join(ids, ",");
    }
    
    private Map<String, String> _getContextVars(String siteName, String language)
    {
        Map<String, String> contextVars = new HashMap<>();
        
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_LANGUAGE, language);
        
        return contextVars;
    }

}
