/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.tagcloud.cache;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This class handles the cache used by the tag cloud plugin
 */
public class TagCloudCacheManager extends AbstractLogEnabled implements Component
{
    /** Avalon role */
    public static final String ROLE = TagCloudCacheManager.class.getName();

    // Cache by workspace and zone item
    private Map<String, Map<String, Object>> _objects = new HashMap<>();

    /**
     * Add tag cloud in cache
     * @param workspaceName The workspace name
     * @param zoneItemId The zone item id
     * @param object The tag cloud object itself
     */
    public void addTagCloud (String workspaceName, String zoneItemId, Object object)
    {
        if (!_objects.containsKey(workspaceName))
        {
            _objects.put(workspaceName, new HashMap<>());
        }
        
        _objects.get(workspaceName).put(zoneItemId, object);
    }

    /**
     * Get the tag cloud in cache or null if it is not in cache.
     * @param workspaceName The workspace name
     * @param zoneItemId The zone item id
     * @return The tag cloud
     */
    public Object getTagCloud (String workspaceName, String zoneItemId)
    {
        if (_objects.containsKey(workspaceName))
        {
            return _objects.get(workspaceName).get(zoneItemId);
        }
        return null;
    }

    /**
     * Determines if the tag cloud is in cache
     * @param workspaceName The workspace name
     * @param zoneItemId The zone item id
     * @return true if the tag cloud is in cache
     */
    public boolean hasTagCloud (String workspaceName, String zoneItemId)
    {
        return _objects.containsKey(workspaceName) && _objects.get(workspaceName).containsKey(zoneItemId);
    }

    /**
     * Invalid tag cloud for all workspaces
     * @param zoneItemId the zone item id
     */
    public void invalidateTagCloud (String zoneItemId)
    {
        for (String workspaceName : _objects.keySet())
        {
            _objects.get(workspaceName).remove(zoneItemId);
        }
    }
    
    /**
     * Invalid cache for all workspaces
     */
    public void invalidateCache ()
    {
        _objects = new HashMap<>();
    }
    
}
