/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.tagcloud.generators;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.xml.sax.SAXException;

import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.cms.search.solr.SolrClientProvider;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.tagcloud.cache.TagCloudCacheManager;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.indexing.solr.SolrWebFieldNames;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.search.query.PageContentQuery;
import org.ametys.web.search.query.PageQuery;

/**
 * Generator for tag clouds
 */
public abstract class AbstractTagCloudGenerator extends ServiceableGenerator implements Contextualizable
{
    
    /** Compares tag cloud items  */
    protected static final Comparator<TagCloudItem> OCCURRENCE_COMPARATOR = new ItemOccurrenceComparator();
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The cache manager */
    protected TagCloudCacheManager _cacheManager;
    
    /** The solr client provider. */
    protected SolrClientProvider _solrClientProvider;
    
    /** The solr client. */
    protected SolrClient _solrClient;
    
    /** The context */
    protected Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _cacheManager = (TagCloudCacheManager) serviceManager.lookup(TagCloudCacheManager.ROLE);
        _solrClientProvider = (SolrClientProvider) serviceManager.lookup(SolrClientProvider.ROLE);
        _solrClient = _solrClientProvider.getReadClient();
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        String siteName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME);
        String lang = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);

        // Fetch the needed parameters from the zone item
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();

        // Content types
        String[] cTypes = serviceParameters.getValue("content-types");

        // Pages
        String[] pages = serviceParameters.getValue("search-by-pages");

        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        if (!_cacheManager.hasTagCloud(currentWsp, zoneItem.getId()))
        {
            List<TagCloudItem> tagCloud = getTagCloudItems(siteName, lang, serviceParameters);
            _cacheManager.addTagCloud(currentWsp, zoneItem.getId(), tagCloud);
        }
        
        @SuppressWarnings("unchecked")
        List<TagCloudItem> tagCloud = (List<TagCloudItem>) _cacheManager.getTagCloud(currentWsp, zoneItem.getId());
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "tagCloud");
        
        if (!tagCloud.isEmpty())
        {
            // Limit the number of items
            long limit = serviceParameters.getValue("limit", false, (long) tagCloud.size()); // Defaults to max number of tags
            int length = (int) limit;

            int max = tagCloud.get(0).getOccurrenceCount();
            int min = tagCloud.size() < length ? tagCloud.get(tagCloud.size() - 1).getOccurrenceCount() : tagCloud.get(length - 1).getOccurrenceCount();
            
            Iterator<TagCloudItem> tagCloudIt = tagCloud.iterator();
            
            int i = 0;
            while (tagCloudIt.hasNext() && i < length)
            {
                saxTagCloudItem(tagCloudIt.next(), min, max);
                i++;
            }
            
            // Search form values
            _saxFormParameters(cTypes, pages);
            
            // Search engine page
            String pageId = serviceParameters.getValue("search-engine-page", false, null);
            if (StringUtils.isNotEmpty(pageId))
            {
                Page page = _resolver.resolveById(pageId);
                XMLUtils.createElement(contentHandler, "searchUrl", page.getSitemapName() + "/" + page.getPathInSitemap());
            }
        }
        
        XMLUtils.endElement(contentHandler, "tagCloud");
        contentHandler.endDocument();
    }
    
    /**
     * Get the tag cloud items
     * @param siteName The site name
     * @param lang The language
     * @param serviceParameters The service parameters
     * @return The list og tag cloud item
     * @throws IOException if an error occurs when manipulating files
     * @throws ProcessingException if an error occurs during the retrieving of the tag cloud items
     */
    protected abstract List<TagCloudItem> getTagCloudItems(String siteName, String lang, ModelAwareDataHolder serviceParameters) throws IOException, ProcessingException;
    
    /**
     * Sax a tag cloud item
     * @param item The tag cloud item to sax
     * @param min The min number of occurrence
     * @param max The max number of occurrence
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxTagCloudItem(TagCloudItem item, int min, int max) throws SAXException
    {
        int nbOccurence = item.getOccurrenceCount();
        int position = item.getPosition();
        int fontSize = _getFontSize(nbOccurence, min, max);
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("nb", String.valueOf(nbOccurence));
        attrs.addCDATAAttribute("original-position", String.valueOf(position));
        attrs.addCDATAAttribute("font-size", String.valueOf(fontSize));
        attrs.addCDATAAttribute("frequency", String.valueOf(fontSize + 1));
        _saxAdditionalAttributes(item, attrs);
        
        XMLUtils.startElement(contentHandler, "item", attrs);
        
        List<String> i18nParams = new ArrayList<>();
        i18nParams.add(String.valueOf(fontSize + 1));
        new I18nizableText("plugin.tagcloud", "PLUGINS_TAGCLOUD_TAGS_SERVICE_FREQUENCY", i18nParams).toSAX(contentHandler, "frequency");
        
        item.getWord().toSAX(contentHandler);
        
        XMLUtils.endElement(contentHandler, "item");
    }
    
    /**
     * Build a {@link SolrQuery} from a Query object.
     * @param query the Query object.
     * @return The SolrQuery.
     * @throws QuerySyntaxException if an error occurs.
     */
    protected SolrQuery build(Query query) throws QuerySyntaxException
    {
        String queryString = query.build();
        
        SolrQuery solrQuery = new SolrQuery(queryString);
        
        solrQuery.addFilterQuery("_documentType:\"" + SolrWebFieldNames.TYPE_PAGE + "\"");
        
        // Don't return any rows: we just need the result count.
        solrQuery.setRows(0);
        
        return solrQuery;
    }
    
    /**
     * Sax additional attributes for item
     * @param item The tag cloud item
     * @param attrs The attributes
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxAdditionalAttributes(TagCloudItem item, AttributesImpl attrs) throws SAXException
    {
        // Nothing
    }
    
    /**
     * Add content types term query to the queries.
     * @param queries The query collection.
     * @param cTypes the content types. Can be empty or null
     */
    protected void _addContentTypeQuery(Collection<Query> queries, String[] cTypes)
    {
        if (cTypes != null && cTypes.length > 0 && !(cTypes.length == 1 && cTypes[0].equals("")))
        {
            queries.add(new PageContentQuery(new ContentTypeQuery(cTypes)));
        }
    }
    
    /**
     * Add pages term query to the queries
     * @param queries The query collection.
     * @param pageIds The page IDs.
     */
    protected void _addPagesQuery(Collection<Query> queries, String[] pageIds)
    {
        if (pageIds != null && pageIds.length > 0 && !(pageIds.length == 1 && pageIds[0].equals("")))
        {
            List<Query> pageQueries = new ArrayList<>();
            
            for (String pageId : pageIds)
            {
                pageQueries.add(new PageQuery(pageId, true));
            }
            
            queries.add(new OrQuery(pageQueries));
        }
    }
    
    /**
     * SAX teh form search criteria
     * @param cTypes the content types
     * @param pages the pages
     * @throws SAXException if an error occurred while SAXing
     */
    protected void _saxFormParameters (String[] cTypes, String[] pages) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "form");
        
        _saxContentTypeCriteria(cTypes);
        _saxSitemapCriteria(pages);
        
        XMLUtils.endElement(contentHandler, "form");
    }

    /**
     * Get the font size
     * @param nb the number of occurrence
     * @param min the min number of occurrence
     * @param max the max number of occurrence
     * @return the font size
     */
    protected int _getFontSize (int nb, int min, int max)
    {
        double p = ((double) nb - (double) min) / ((double) max - (double) min) * 100.0;
        double interval = 100.0 / 5.0; // 6 sizes
        
        return (int) Math.floor(p / interval);
    }

    
    /**
     * SAX the content types criteria
     * @param cTypes the content types
     * @throws SAXException if an error occurred while SAXing
     */
    protected void _saxContentTypeCriteria (String[] cTypes) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "content-types");
        
        if (cTypes != null && cTypes.length > 0 && !(cTypes.length == 1 && cTypes[0].equals("")))
        {
            for (String cTypeId : cTypes)
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("id", cTypeId);
                XMLUtils.createElement(contentHandler, "type", attr);
            }
        }
        
        XMLUtils.endElement(contentHandler, "content-types");
    }
    
    /**
     * SAX the pages criteria
     * @param pages the pages
     * @throws SAXException if an error occurred while SAXing
     */
    protected void _saxSitemapCriteria (String[] pages)  throws SAXException
    {
        XMLUtils.startElement(contentHandler, "pages");
        
        if (pages != null && pages.length > 0 && !(pages.length == 1 && pages[0].equals("")))
        {
            for (String pageID : pages)
            {
                Page page = _resolver.resolveById(pageID);
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("path", page.getSitemap().getName() + "/" + page.getPathInSitemap());
                attr.addCDATAAttribute("title", page.getTitle());
                attr.addCDATAAttribute("id", pageID);
                XMLUtils.createElement(contentHandler, "page", attr);
            }
        }
        
        XMLUtils.endElement(contentHandler, "pages");
    }
    
//    /**
//     * Returns the analyzer to use
//     * @param language The current language
//     * @return The {@link Analyzer}
//     */
//    protected Analyzer getAnalyzer (String language)
//    {
//        if (language.equals("br"))
//        {
//            return new BrazilianAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("cn"))
//        {
//            return new ChineseAnalyzer();
//        }
//        else if (language.equals("cz"))
//        {
//            return new CzechAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("gr"))
//        {
//            return new GreekAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("de"))
//        {
//            return new GermanAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("fr"))
//        {
//            return new FrenchAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("nl"))
//        {
//            return new DutchAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("ru"))
//        {
//            return new RussianAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else if (language.equals("th"))
//        {
//            return new ThaiAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//        else
//        {
//            //Default to standard analyzer (works well for english)
//            return new StandardAnalyzer(LuceneConstants.LUCENE_VERSION);
//        }
//    }
    
//    /**
//     * Get the index searcher
//     *
//     * @param siteName The site to search in.
//     * @param lang The current language
//     * @return The index searcher
//     * @throws IOException If an error occurred while opening indexes
//     */
//    protected Searcher getSearchIndex(String siteName, String lang) throws IOException
//    {
//        File indexDir = IndexerHelper.getIndexDir(_context, siteName, lang);
//        if (indexDir.exists() && indexDir.listFiles().length > 0)
//        {
//            return new IndexSearcher(FSDirectory.open(indexDir));
//        }
//        else
//        {
//            return null;
//        }
//    }
    
    /**
     * Abstract class for a tag cloud item
     *
     */
    protected interface TagCloudItem
    {
        /**
         * Get the tag cloud word to display
         * @return The tag cloud word
         */
        I18nizableText getWord();
        
        /**
         * Returns the number of occurrence
         * @return the number of occurrence
         */
        int getOccurrenceCount();
        
        /**
         * Get the original position.
         * @return the original position.
         */
        int getPosition();
    }
    
    /**
     * Compares two terms by descending occurrence count.
     */
    protected static class ItemOccurrenceComparator implements Comparator<TagCloudItem>
    {
        
        @Override
        public int compare(TagCloudItem tc1, TagCloudItem tc2)
        {
            return tc2.getOccurrenceCount() - tc1.getOccurrenceCount();
        }
        
    }
    
}
