/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.tagcloud.generators;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.xml.sax.SAXException;

import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.TagQuery;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.search.query.PageContentQuery;
import org.ametys.web.search.query.SiteQuery;
import org.ametys.web.search.query.SitemapQuery;

/**
 * Generates a tag cloud based upon content tags
 */
public class TagCloudOnTagsGenerator extends AbstractTagCloudGenerator
{
    /** The {@link TagProviderExtensionPoint} */
    protected TagProviderExtensionPoint _tagExtPt;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _tagExtPt = (TagProviderExtensionPoint) manager.lookup(TagProviderExtensionPoint.ROLE);
    }

    @Override
    protected List<TagCloudItem> getTagCloudItems(String siteName, String lang, ModelAwareDataHolder serviceParameters) throws IOException, ProcessingException
    {
        // Set of {@link TagCloud} sorted by occurrence
        List<TagCloudItem> tagCloud = new ArrayList<>();

        // Content types
        String[] cTypes = serviceParameters.getValue("content-types");

        // Pages
        String[] pages = serviceParameters.getValue("search-by-pages");
        
        // Tags to search for
        String[] tagArray = serviceParameters.getValue("search-by-tags");
        List<Tag> tags = _getTags(siteName, tagArray);
        
        int pos = 0;
        for (Tag tag : tags)
        {
            try
            {
                Query queryObject = getQuery(siteName, lang, tag.getName(), cTypes, pages);
                
                SolrQuery query = build(queryObject);
                
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("Solr query: " + URIUtils.decode(query.toString()));
                }
                
                String collection = _solrClientProvider.getCollectionName();
                QueryResponse response = _solrClient.query(collection, query);
                
                int count = (int) response.getResults().getNumFound();
                if (count > 0)
                {
                    tagCloud.add(new TagTagCloudItem(count, tag, pos));
                    pos++;
                }
            }
            catch (Exception e)
            {
                getLogger().error("Query on tag " + tag.getName() + " failed. Unable to get number of occurrence", e);
                throw new ProcessingException("Query on tag " + tag.getName() + " failed. Unable to get number of occurrence", e);
            }
        }
        
        Collections.sort(tagCloud, OCCURRENCE_COMPARATOR);
        
        return tagCloud;
    }
    
    @Override
    protected void _saxAdditionalAttributes(TagCloudItem item, AttributesImpl attrs) throws SAXException
    {
        if (item instanceof TagTagCloudItem)
        {
            Tag tag = ((TagTagCloudItem) item).getTag();
            attrs.addCDATAAttribute("tag", tag.getName());
        }
    }
    
    /**
     * Get the query
     * @param siteName The site name.
     * @param language The current language.
     * @param tagName The tag name
     * @param contentTypes The content types
     * @param pages The pages
     * @return the query
     * @throws IllegalArgumentException If the search field is invalid
     */
    protected Query getQuery(String siteName, String language, String tagName, String[] contentTypes, String[] pages)
    {
        List<Query> queries = new ArrayList<>();
        
        Query siteQuery = new SiteQuery(siteName);
        Query langQuery = new SitemapQuery(language);
        
        queries.add(siteQuery);
        queries.add(langQuery);
        
        _addContentTypeQuery(queries, contentTypes);
        
        _addPagesQuery(queries, pages);
        
        _addTagQuery(queries, tagName);
        
        return new AndQuery(queries);
    }
    
    private void _addTagQuery(Collection<Query> queries, String tagName)
    {
        queries.add(new PageContentQuery(new TagQuery(tagName)));
    }
    
    private List<Tag> _getTags(String siteName, String[] tagArray)
    {
        Map<String, Object> contextParameters = new HashMap<>();
        contextParameters.put("siteName", siteName);
        
        List<Tag> tags = new ArrayList<>();
        
        for (String tagName : tagArray)
        {
            boolean found = false;
            Iterator<String> extensionsIds = _tagExtPt.getExtensionsIds().iterator();
            
            while (extensionsIds.hasNext() && !found)
            {
                String id = extensionsIds.next();
                
                TagProvider<CMSTag> tagProvider = _tagExtPt.getExtension(id);
                if (tagName.startsWith("provider_") && id.equals(tagName.substring("provider_".length())))
                {
                    found = true;
                    tags.addAll(tagProvider.getTags(contextParameters).values());
                }
                else if (tagProvider.hasTag(tagName, contextParameters))
                {
                    found = true;
                    Tag tag = tagProvider.getTag(tagName, contextParameters);
                    tags.addAll(tag.getTags().values());
                }
            }
        }
        
        return tags;
    }
    
    /**
     * Class representing an item of tag cloud
     */
    private class TagTagCloudItem implements TagCloudItem
    {
        private int _occurence;
        private Tag _tag;
        private int _position;

        /**
         * Constructor
         * 
         * @param occurence the number of occurence
         * @param tag the tag
         * @param position the item original position.
         */
        public TagTagCloudItem(int occurence, Tag tag, int position)
        {
            _occurence = occurence;
            _tag = tag;
            _position = position;
        }

        /**
         * Returns the tag
         * 
         * @return the tag
         */
        public Tag getTag()
        {
            return _tag;
        }

        @Override
        public I18nizableText getWord()
        {
            return _tag.getTitle();
        }
        
        @Override
        public int getOccurrenceCount()
        {
            return _occurence;
        }
        
        @Override
        public int getPosition()
        {
            return _position;
        }
    }
    
}
