/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.tagcloud.generators;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.response.QueryResponse;

import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.search.query.PageContentQuery;
import org.ametys.web.search.query.SiteQuery;
import org.ametys.web.search.query.SitemapQuery;

/**
 * Generates a tag cloud based upon words
 */
public class TagCloudOnWordsGenerator extends AbstractTagCloudGenerator
{
    
    @Override
    protected List<TagCloudItem> getTagCloudItems(String siteName, String lang, ModelAwareDataHolder serviceParameters) throws IOException, ProcessingException
    {
        // Content types
        String[] cTypes = serviceParameters.getValue("content-types");

        // Pages
        String[] pages = serviceParameters.getValue("search-by-pages");
        
        // Get key words to match. 
        // TODO If empty, use top ranking terms from the solr server.
        String keywordsAsString = serviceParameters.getValue("keywords");
        String[] keywords = keywordsAsString.split("[,;\n]");
        
        // Set of {@link TagCloudItem} sorted by occurrence
        List<TagCloudItem> tagCloud = new ArrayList<>();
        
        int pos = 0;
        for (String keyword : keywords)
        {
            String trimKeyword = StringUtils.trimToNull(keyword);
            if (trimKeyword != null)
            {
                try
                {
                    Query queryObject = getQuery(siteName, lang, trimKeyword, cTypes, pages);
                    
                    SolrQuery query = build(queryObject);
                    
                    if (getLogger().isInfoEnabled())
                    {
                        getLogger().info("Solr query: " + URIUtils.decode(query.toString()));
                    }
                    
                    String collection = _solrClientProvider.getCollectionName();
                    QueryResponse response = _solrClient.query(collection, query);
                    
                    int count = (int) response.getResults().getNumFound();
                    if (count > 0)
                    {
                        tagCloud.add(new WordTagCloudItem(count, trimKeyword, pos));
                        pos++;
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("Query on keyword '" + trimKeyword + "' failed. Unable to get number of occurrence", e);
                    throw new ProcessingException("Query on keyword '" + trimKeyword + "' failed. Unable to get number of occurrence", e);
                }
            }
        }
        
        Collections.sort(tagCloud, OCCURRENCE_COMPARATOR);
        
        return tagCloud;
    }
    
    /**
     * Get the query
     * @param siteName The site name.
     * @param language The current language.
     * @param keyword The key word
     * @param contentTypes The content types
     * @param pages The pages
     * @return the query object.
     * @throws IllegalArgumentException If the search field is invalid
     */
    protected Query getQuery(String siteName, String language, String keyword, String[] contentTypes, String[] pages) throws IllegalArgumentException
    {
        List<Query> queries = new ArrayList<>();
        
        Query siteQuery = new SiteQuery(siteName);
        Query langQuery = new SitemapQuery(language);
        
        queries.add(siteQuery);
        queries.add(langQuery);
        
        _addContentTypeQuery(queries, contentTypes);
        
        _addPagesQuery(queries, pages);
        
        _addTextFieldQuery(queries, language, keyword);
        
        return new AndQuery(queries);
    }
    
    // One or more of the words
    private void _addTextFieldQuery(Collection<Query> queries, String language, String keyword) throws IllegalArgumentException
    {
        if (StringUtils.isNotBlank(keyword))
        {
            Query query = new FullTextQuery(keyword, language, Operator.SEARCH);
            Query contentQuery = new PageContentQuery(query);
            
            queries.add(new OrQuery(query, contentQuery));
        }
    }
    
    private class WordTagCloudItem implements TagCloudItem
    {
        private int _occurrence;
        private String _keyword;
        private int _position;
        
        /**
         * Constructor
         * 
         * @param occurrence the number of occurrence
         * @param keyword the keyword
         * @param position the original position
         */
        public WordTagCloudItem(int occurrence, String keyword, int position)
        {
            _occurrence = occurrence;
            _keyword = keyword;
            _position = position;
        }
        
        @Override
        public I18nizableText getWord()
        {
            return new I18nizableText(_keyword);
        }
        
        @Override
        public int getOccurrenceCount()
        {
            return _occurrence;
        }
        
        @Override
        public int getPosition()
        {
            return _position;
        }
    }

}

