/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.clientsideelement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableDataHolder;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.content.jcr.DefaultWebContent;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinsManager;

import com.opensymphony.workflow.WorkflowException;

/**
 * Client side element for UGC content moderation
 *
 */
public class UGCContentModerationClientSideElement extends StaticClientSideElement
{
    /** The Ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** The page DAO */
    protected PageDAO _pageDAO;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The skins manager */
    protected SkinsManager _skinsManager;
    /** The observation manager */
    protected ObservationManager _observationManager;
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;

    @Override   
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _pageDAO = (PageDAO) smanager.lookup(PageDAO.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    /**
     * Accept a UGC content
     * @param contentIds The id of UGC contents
     * @param targetContentType The id of target content type. Can be null or empty to not change content type
     * @param targetWorkflowName The workflow name for contents to create
     * @param initActionId The id of workflow init action
     * @param mode The insertion mode ('new' to insert contents in a new page, 'affect' to insert contents on a existing page or 'none' to keep content as orphan).
     * @param pageId The page id. Can be null for mode 'none'
     * @return the result
     * @throws ProcessingException if failed to transform UGC contents
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> acceptUGCContent(List<String> contentIds, String targetContentType, String targetWorkflowName, int initActionId, String mode, String pageId) throws ProcessingException
    {
        Map<String, String> rights = getRights(Map.of());
        if (rights.isEmpty())
        {
            throw new IllegalStateException("UGC moderation action cannot be called from a non-protected controller '" + getId() + "'. Right configuration is required.");
        }
        
        if (!hasRight(rights))
        {
            throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " try to access privileges feature without sufficient rights");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        result.put("createdContents", new ArrayList<>());
        result.put("deletedContents", new HashSet<>());
        
        for (String contentId : contentIds)
        {
            try
            {
                DefaultWebContent ugcContent = _resolver.resolveById(contentId);
                
                // Create the content
                Content newContent = createContent(ugcContent, targetWorkflowName, initActionId, targetContentType);
                
                try
                {
                    // Copy data
                    ugcContent.copyTo((ModifiableDataHolder) newContent);
                }
                catch (UndefinedItemPathException e)
                {
                    getLogger().warn("The target content type '{}' is not compatible with the source UGC content type '{}'", targetContentType, ugcContent.getTypes(), e);
                    
                    result.put("success", false);
                    result.put("error", "invalid-content-type");
                    return result;
                }
                
                // Save changes
                ((ModifiableAmetysObject) newContent).saveChanges();
                
                // Notify observers
                Map<String, Object> contentEventParams = new HashMap<>();
                contentEventParams.put(ObservationConstants.ARGS_CONTENT, newContent);
                contentEventParams.put(ObservationConstants.ARGS_CONTENT_ID, newContent.getId());
                contentEventParams.put(ObservationConstants.ARGS_CONTENT_NAME, newContent.getName());
                _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_MODIFIED, _currentUserProvider.getUser(), contentEventParams));
                
                // Create a new version
                ((VersionableAmetysObject) newContent).checkpoint();
                
                ModifiablePage page = null;
                if ("new".equals(mode))
                {
                    // Create new page to insert content
                    page = createPage (pageId, ugcContent);
                }
                else if ("affect".equals(mode))
                {
                    // Insert content on given page
                    page = _resolver.resolveById(pageId);
                }
                
                String zoneName = null;
                if (page != null)
                {
                    zoneName = getZoneName (page);
                    if (zoneName == null)
                    {
                        getLogger().warn("Selected page '{}' is not a container page: can not affect a content", pageId);
                        
                        result.put("success", false);
                        result.put("error", "invalid-page");
                        return result;
                    }
                    
                    ModifiableZone zone = null;
                    if (page.hasZone(zoneName))
                    {
                        zone = page.getZone(zoneName);
                    }
                    else
                    {
                        zone = page.createZone(zoneName);
                    }
                    
                    ModifiableZoneItem zoneItem = zone.addZoneItem();
                    zoneItem.setType(ZoneType.CONTENT);
                    zoneItem.setContent(newContent);
                    zoneItem.setViewName("main");
                    
                    page.saveChanges();
                    
                    Map<String, Object> pageEventParams = new HashMap<>();
                    pageEventParams.put(org.ametys.web.ObservationConstants.ARGS_SITEMAP_ELEMENT, page);
                    pageEventParams.put(org.ametys.web.ObservationConstants.ARGS_ZONE_ITEM_ID, zoneItem.getId());
                    pageEventParams.put(org.ametys.web.ObservationConstants.ARGS_ZONE_TYPE, ZoneType.CONTENT);
                    pageEventParams.put(org.ametys.web.ObservationConstants.ARGS_ZONE_ITEM_CONTENT, newContent);
                    _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_ZONEITEM_ADDED, _currentUserProvider.getUser(), pageEventParams));
                }
                
                Map<String, Object> contentInfo = new HashMap<>();
                contentInfo.put("title", newContent.getTitle());
                contentInfo.put("id", newContent.getId());
                if (page != null)
                {
                    contentInfo.put("pageId", page.getId());
                }
                
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> acceptedContents = (List<Map<String, Object>>) result.get("createdContents");
                acceptedContents.add(contentInfo);
                
                // Notify observers
                _observationManager.notify(new Event(org.ametys.plugins.ugc.observation.ObservationConstants.EVENT_UGC_CONTENT_ACCEPTED, _currentUserProvider.getUser(), contentEventParams));
                
                // Delete initial content
                String ugcContentId = ugcContent.getId();
                deleteContent(ugcContent);
                @SuppressWarnings("unchecked")
                Set<String> deletedContents = (Set<String>) result.get("deletedContents");
                deletedContents.add(ugcContentId);
            }
            catch (WorkflowException | AmetysRepositoryException e)
            {
                getLogger().error("Unable to transform UGC content '" + contentId + "'", e);
                throw new ProcessingException("Unable to transform UGC content '" + contentId + "'", e);
            }
        }
        
        result.put("success", true);
        return result;
    }
    
    /**
     * Refuse UGC content
     * @param contentIds The id of UGC contents to refuse
     * @param comment The reject's comment
     * @param withNotification True to notify UGC author of the refuse
     * @return the result
     * @throws ProcessingException if failed to transform UGC contents
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseUGCContent(List<String> contentIds, String comment, boolean withNotification) throws ProcessingException
    {
        Map<String, String> rights = getRights(Map.of());
        if (rights.isEmpty())
        {
            throw new IllegalStateException("UGC moderation action cannot be called from a non-protected controller '" + getId() + "'. Right configuration is required.");
        }
        
        if (!hasRight(rights))
        {
            throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " try to access privileges feature without sufficient rights");
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("deletedContents", new HashSet<>());
        
        for (String contentId : contentIds)
        {
            try
            {
                DefaultWebContent ugcContent = _resolver.resolveById(contentId);
                
                // Notify observers
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_CONTENT, ugcContent);
                eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, ugcContent.getName());
                eventParams.put(ObservationConstants.ARGS_CONTENT_ID, ugcContent.getId());
                eventParams.put(org.ametys.plugins.ugc.observation.ObservationConstants.ARGS_UGC_REFUSE_NOTIFY, withNotification);
                eventParams.put(org.ametys.plugins.ugc.observation.ObservationConstants.ARGS_UGC_REFUSE_COMMENT, comment);
                
                _observationManager.notify(new Event(org.ametys.plugins.ugc.observation.ObservationConstants.EVENT_UGC_CONTENT_REFUSED, _currentUserProvider.getUser(), eventParams));
                
                // Delete initial content
                String ugcContentId = ugcContent.getId();
                deleteContent(ugcContent);
                @SuppressWarnings("unchecked")
                Set<String> deletedContents = (Set<String>) result.get("deletedContents");
                deletedContents.add(ugcContentId);
            }
            catch (AmetysRepositoryException e)
            {
                getLogger().error("Unable to refuse UGC content '" + contentId + "'", e);
                throw new ProcessingException("Unable to refuse UGC content '" + contentId + "'", e);
            }
        }
        
        result.put("success", true);
        return result;
    }
    
    /**
     * Create page under a parent page
     * @param parentId the parent page id or empty for the sitemap root
     * @param content the UGC content
     * @return the new created page
     */
    protected ModifiablePage createPage (String parentId, DefaultWebContent content)
    {
        String realParentId = parentId;
        if (StringUtils.isEmpty(parentId))
        {
            Site site = _siteManager.getSite(content.getSiteName());
            realParentId = site.getSitemap(content.getLanguage()).getId();
        }

        Map<String, Object> result = _pageDAO.createPage(realParentId, content.getTitle(), "");
        ModifiablePage page = _resolver.resolveById((String) result.get("id"));
        
        _pageDAO.setTemplate(Collections.singletonList(page.getId()), "page");
        
        return page;
    }
    
    /**
     * Get the name of zone where to insert content
     * @param page The page
     * @return the zone's name
     */
    protected String getZoneName (Page page)
    {
        if (page.getType() == PageType.CONTAINER)
        {
            String skinId = page.getSite().getSkinId();
            Skin skin = _skinsManager.getSkin(skinId);
            SkinTemplate template = skin.getTemplate(page.getTemplate());
            
            // Has a default zone ?
            if (template.getZone("default") != null)
            {
                return "default";
            }
            else
            {
                return template.getZones().keySet().iterator().next();
            }
        }
        return null;
    }
    
    /**
     * Create the content from the proposed content
     * @param initialContent The initial content
     * @param workflowName the workflow name
     * @param actionId The init action id
     * @param cTypeId the content type
     * @return the created content
     * @throws WorkflowException if failed to create content
     */
    protected Content createContent (Content initialContent, String workflowName, int actionId, String cTypeId) throws WorkflowException
    {
        Map<String, Object> inputs = new HashMap<>();
        inputs.put("prevent-version-creation", "true");
        
        if (initialContent instanceof WebContent)
        {
            inputs.put(org.ametys.web.workflow.CreateContentFunction.SITE_KEY, ((WebContent) initialContent).getSiteName());
        }
        
        Map<String, Object> result =  _contentWorkflowHelper.createContent(workflowName,
                actionId, 
                initialContent.getName(), 
                initialContent.getTitle(), 
                new String[] {cTypeId}, 
                new String[] {UGCConstants.UGC_MIXIN_TYPE}, 
                initialContent.getLanguage(), 
                inputs);
        
        return (Content) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
        
    }
    
    /**
     * Delete the content
     * @param content the content to delete
     */
    protected void deleteContent(Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, content.getName());
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETING, _currentUserProvider.getUser(), eventParams));
        
        RemovableAmetysObject removableContent = (RemovableAmetysObject) content;
        ModifiableAmetysObject parent = removableContent.getParent();
        
        // Remove the content.
        removableContent.remove();
        
        parent.saveChanges();
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETED, _currentUserProvider.getUser(), eventParams));
    }
}
