/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.observation;

import java.io.IOException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.FormatStyle;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableDateTime;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

import jakarta.mail.MessagingException;

/**
 * Abstract observer to send mail to UGC author on UGC content event
 *
 */
public abstract class AbstractUGCContentObserver extends AbstractLogEnabled implements Observer, Serviceable, Configurable
{
    /** The content type helper */
    protected ContentTypesHelper _cTypeHelper;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The user manager */
    protected UserManager _userManager;
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;
    
    /** The i18n key for mail subject */
    protected String _subjectI18nKey;
    /** The i18n key for mail body */
    protected String _bodyI18nKey;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _userLanguagesManager = (UserLanguagesManager) smanager.lookup(UserLanguagesManager.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _subjectI18nKey = configuration.getChild("mailSubjectKey").getValue(null);
        _bodyI18nKey = configuration.getChild("mailBodyKey").getValue(null);
    }
    
    @Override
    public int getPriority()
    {
        return MIN_PRIORITY;
    }
    
    /**
     * Send mail to the UGC author
     * @param ugcContent The UGC content
     * @param comment The comment. Can be null.
     */
    protected void sendMailToUGCAuthor(Content ugcContent, String comment)
    {
        String recipient = ugcContent.getValue(UGCConstants.ATTRIBUTE_PATH_UGC_MAIL, false, null);
        if (StringUtils.isNotBlank(recipient))
        {
            String language = getMailLanguage(ugcContent);
            String sender = Config.getInstance().getValue("smtp.mail.from");
            if (ugcContent instanceof WebContent)
            {
                Site site = ((WebContent) ugcContent).getSite();
                sender = site.getValue("site-mail-from", false, Config.getInstance().getValue("smtp.mail.from"));
            }
            
            String subject = getMailSubject(ugcContent, language);
            String body = getMailBody(ugcContent, comment, language);
            
            try
            {
                SendMailHelper.newMail()
                              .withSubject(subject)
                              .withTextBody(body)
                              .withSender(sender)
                              .withRecipient(recipient)
                              .withAsync(true)
                              .sendMail();
            }
            catch (MessagingException | IOException e)
            {
                getLogger().warn("Fail to send UGC content notification mail to " + recipient, e);
            }
        }
    }
    
    /**
     * Get the mail language
     * @param ugcContent The UGC content
     * @return the language to use in the mail or null if none was found
     */
    protected String getMailLanguage(Content ugcContent)
    {
        String language = null;
        UserIdentity userIdentity = ugcContent.getCreator();
        if (userIdentity != null && !UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(userIdentity))
        {
            User user = _userManager.getUser(userIdentity);
            if (user != null)
            {
                language = user.getLanguage();
            }
        }
        
        // ugc content language can also be null
        return StringUtils.defaultIfEmpty(language, _userLanguagesManager.getDefaultLanguage());
    }
    
    /**
     * Get the mail subject
     * @param ugcContent The UGC content
     * @param language The language to use
     * @return the mail text subject
     */
    protected String getMailSubject (Content ugcContent, String language)
    {
        I18nizableText subjectKey = new I18nizableText(null, _subjectI18nKey, getSubjectI18nParams(ugcContent));
        return _i18nUtils.translate(subjectKey, language);
    }
    
    /**
     * Get the mail body
     * @param ugcContent The UGC content
     * @param comment The comment. Can be null.
     * @param language The language to use
     * @return the mail text body
     */
    protected String getMailBody (Content ugcContent, String comment, String language)
    {
        I18nizableText bodyKey = new I18nizableText(null, _bodyI18nKey, getBodyI18nParams(ugcContent, comment));
        return _i18nUtils.translate(bodyKey, language);
    }
    
    /**
     * Get the i18n parameters for mail subject
     * @param ugcContent The UGC content
     * @return the i18n parameters for mail subject
     */
    protected Map<String, I18nizableTextParameter> getSubjectI18nParams(Content ugcContent)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        
        params.put("contentTitle", new I18nizableText(ugcContent.getTitle()));
        if (ugcContent instanceof WebContent)
        {
            Site site = ((WebContent) ugcContent).getSite();
            params.put("siteTitle", new I18nizableText(site.getTitle()));
        }
        
        return params;
    }
    
    /**
     * Get the i18n parameters for mail body
     * @param ugcContent The UGC content
     * @param comment The comment. Can be null.
     * @return the i18n parameters for mail body
     */
    protected Map<String, I18nizableTextParameter> getBodyI18nParams(Content ugcContent, String comment)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        
        String authorFullName = ugcContent.getValue(UGCConstants.ATTRIBUTE_PATH_UGC_AUTHOR, false, null);
        params.put("authorFullName", new I18nizableText(authorFullName));
        
        params.put("contentTitle", new I18nizableText(ugcContent.getTitle()));
        
        if (ugcContent instanceof WebContent)
        {
            Site site = ((WebContent) ugcContent).getSite();
            params.put("siteTitle", new I18nizableText(site.getTitle()));
            params.put("siteUrl", new I18nizableText(site.getUrl()));
            
            Collection<Page> pages = ((WebContent) ugcContent).getReferencingPages();
            if (!pages.isEmpty())
            {
                Page page = pages.iterator().next();
                String uri = site.getUrl() + "/" + page.getSitemapName() + "/" + page.getPathInSitemap() + ".html";
                params.put("uri", new I18nizableText(uri));
            }
        }
        
        ZonedDateTime creationDate = ugcContent.getCreationDate();
        params.put("creationDate", new I18nizableDateTime(creationDate, ZoneId.from(creationDate), FormatStyle.LONG));
        
        if (StringUtils.isNotBlank(comment))
        {
            params.put("comment", new I18nizableText(comment));
        }
        
        return params;
    }
}
