/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.workflow;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.workflow.SendMailFunction;

import com.opensymphony.workflow.WorkflowException;

/**
 * OS workflow function to send mail after an action is triggered.
 */
public class SendMailToCreatorFunction extends SendMailFunction
{
    private ContentTypesHelper _cTypeHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    protected Map<String, Set<String>> getRecipientsByLanguage(Map transientVars, WorkflowAwareContent content, Set<String> rights) throws WorkflowException
    {
        String language = null;
        String email = null;
        
        UserIdentity creator = content.getCreator();
        if (!UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(creator))
        {
            User user = _userManager.getUser(creator);
            language = user.getLanguage();
            email = user.getEmail();
        }
        else if (_cTypeHelper.isInstanceOf(content, UGCConstants.UGC_MIXIN_TYPE))
        {
            email = content.getValue(UGCConstants.ATTRIBUTE_PATH_UGC_MAIL, false, null);
        }
        
        language = StringUtils.defaultIfBlank(language, _userLanguagesManager.getDefaultLanguage());
        
        return StringUtils.isNotBlank(email) ? Collections.singletonMap(language, Set.of(email)) : Collections.EMPTY_MAP;
    }
    
    @Override
    protected String getSender(Map transientVars, WorkflowAwareContent content) throws WorkflowException
    {
        if (content instanceof WebContent)
        {
            Site site = ((WebContent) content).getSite();
            return site.getValue("site-mail-from", false, Config.getInstance().getValue("smtp.mail.from"));
        }
        else
        {
            return Config.getInstance().getValue("smtp.mail.from");
        }
    }
    
    @Override
    protected List<String> getBodyI18nParams(User user, WorkflowAwareContent content)
    {
        List<String> params = new ArrayList<>();
        
        params.add(user != null ? user.getFullName() : ""); // {0}
        params.add(_contentHelper.getTitle(content)); // {1}
        
        Site site = _getSite(content);
        if (site != null)
        {
            params.add(site.getTitle());  // {2}
        }
        
        params.add(_getContentUri(content, null, site));  // {3}
        
        // Creator {4}
        if (user != null && !UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user.getIdentity()))
        {
            params.add(user.getFullName());
        }
        else if (_cTypeHelper.isInstanceOf(content, UGCConstants.UGC_MIXIN_TYPE) && content.hasValue(UGCConstants.ATTRIBUTE_PATH_UGC_MAIL))
        {
            params.add(content.getValue(UGCConstants.ATTRIBUTE_PATH_UGC_AUTHOR));
        }
        
        return params;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.ugc", "PLUGINS_UGC_SEND_MAIL_TO_CREATOR_FUNCTION_LABEL");
    }
}
