/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.cachepolicy;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.core.observation.Event;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.web.cache.pageelement.PageElementCachePolicy;
import org.ametys.web.inputdata.SitemapInputData;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Cache policy for the sitemap, handling UGC virtual pages linked with contents.
 * Used for the Sitemap InputData as well as for the Sitemap service, even if the page element cache is not the same.
 */
public class UGCVirtualPagesCachePolicy extends AbstractLogEnabled implements Serviceable, PageElementCachePolicy
{
    private static final Set<String> _UGC_PE_TYPES = new HashSet<>();
    static
    {
        // Used for the Sitemap InputData as well as for the Sitemap service.
        _UGC_PE_TYPES.add(SitemapInputData.class.getName());
        _UGC_PE_TYPES.add("SERVICE:org.ametys.web.service.SitemapService");
    }
    
    /** The UGC page handler */
    protected UGCPageHandler _ugcPageHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _ugcPageHandler = (UGCPageHandler) manager.lookup(UGCPageHandler.ROLE);
    }
    
    @Override
    public Set<String> getPageElementTypes()
    {
        return _UGC_PE_TYPES;
    }
    
    @Override
    public final PolicyResult shouldClearCache(String workspace, Site site, String pageElementType, Event event)
    {
        String id = event.getId();
        
        if (_getRemovingCacheEventIds(workspace).contains(id))
        {
            Object object = event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            
            // The target must be a UGC content and the site must possess a UGC root
            // in one of its sitemaps to be invalidated.
            if (object instanceof ModifiableDefaultContent)
            {
                ModifiableDefaultContent content = (ModifiableDefaultContent) object;
                try
                {
                    for (String contentTypeId : content.getTypes())
                    {
                        for (Sitemap sitemap : site.getSitemaps())
                        {
                            if (_ugcPageHandler.getUGCRootPage(site.getName(), sitemap.getSitemapName(), contentTypeId) != null)
                            {
                                return PolicyResult.REMOVE;
                            }
                            
                        }
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("An error occurred with an event on content " + content.getId(), e);
                    return PolicyResult.KEEP;
                }
            }
        }
        
        return PolicyResult.KEEP;
    }
    
    @Override
    public final PolicyResult shouldClearCache(String workspace, Site site, String pageElementType, String elementId, Event event)
    {
        // Never called because the first-level method never returns NEED_INFORMATION.
        throw new UnsupportedOperationException("Should never be called.");
    }
    
    /**
     * Returns all event ids for which the cache should be removed.
     * @param workspace the current JCR workspace.
     * @return all event ids for which the cache should be removed.
     */
    protected List<String> _getRemovingCacheEventIds(String workspace)
    {
        if ("default".equals(workspace))
        {
            return Arrays.asList(ObservationConstants.EVENT_CONTENT_ADDED,
                                 ObservationConstants.EVENT_CONTENT_MODIFIED,
                                 ObservationConstants.EVENT_CONTENT_DELETED);
        }
        else if ("live".equals(workspace))
        {
            return Arrays.asList(ObservationConstants.EVENT_CONTENT_ADDED,
                                 ObservationConstants.EVENT_CONTENT_VALIDATED,
                                 ObservationConstants.EVENT_CONTENT_DELETED,
                                 ObservationConstants.EVENT_CONTENT_UNTAG_LIVE);
        }
        
        return Collections.emptyList();
    }
}
