/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.generators;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.Optional;

import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.web.WebConstants;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.content.GetSiteAction;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;
import org.ametys.web.synchronization.SynchronizeComponent;

/**
 * UGC service generator
 */
public class UGCGenerator extends ServiceableGenerator
{
    /** The {@link ContentType} manager */
    protected ContentTypeExtensionPoint _cTypeExtPt;
    /** The FO content creation helper */
    protected FOContentCreationHelper _foContentCreationHelper;
    /** The page helper */
    protected PageHelper _pageHelper;
    /** The content helper */
    protected ContentHelper _contentHelper;
    /** The source resolver */
    protected SourceResolver _srcResolver;
    /** The skin manager */
    protected SkinsManager _skinManager;
    /** The content repository */
    protected Repository _repository;
    /** The synchronize component */
    protected SynchronizeComponent _synchronizeComponent;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cTypeExtPt = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _foContentCreationHelper = (FOContentCreationHelper) smanager.lookup(FOContentCreationHelper.ROLE);
        _pageHelper = (PageHelper) smanager.lookup(PageHelper.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _skinManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _repository = (Repository) smanager.lookup(Repository.class.getName());
        _synchronizeComponent = (SynchronizeComponent) smanager.lookup(SynchronizeComponent.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        Page page = (Page) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE);
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String cType = (String) serviceParameters.getValue("content-type");
        if (StringUtils.isBlank(cType))
        {
            throw new IllegalArgumentException("A content type must be defined");
        }
        
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "ugc");
        
        DefinitionContext definitionContext = DefinitionContext.newInstance()
                                                               .withItemTagName("metadata");
        
        // SAX inputs for UGC mixin
        XMLUtils.startElement(contentHandler, "mixin");
        ContentType ugcMixin = _cTypeExtPt.getExtension(UGCConstants.UGC_MIXIN_TYPE);
        ugcMixin.getView("main").toSAX(contentHandler, definitionContext);
        XMLUtils.endElement(contentHandler, "mixin");
        
        // SAX view
        XMLUtils.startElement(contentHandler, "content-type");
        ContentType contentType = _cTypeExtPt.getExtension(cType);
        contentType.getView("main").toSAX(contentHandler, definitionContext);
        XMLUtils.endElement(contentHandler, "content-type");
        
        // SAX contents values for metadata of type CONTENT
        _foContentCreationHelper.saxContentValues(contentHandler, contentType, "items", page.getSitemapName());
        
        XMLUtils.createElement(contentHandler, "has-captcha", String.valueOf(_pageHelper.isCaptchaRequired(page)));
        
        _saxWarnings(zoneItem);
        
        _saxGTUIfNeeded(zoneItem, request);
        
        XMLUtils.endElement(contentHandler, "ugc");
        
        contentHandler.endDocument();
    }
    
    private void _saxWarnings(ZoneItem zoneItem) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "warnings");
        
        String gtuMode = zoneItem.getServiceParameters().getValue("general-terms-of-use-mode", false, "NONE");
        switch (gtuMode)
        {
            case "PAGE":
                Optional<Page> gtuPage = _getGTUPage(zoneItem);
                if (gtuPage.isEmpty() || !_isValid(gtuPage.get()))
                {
                    XMLUtils.startElement(contentHandler, "warning");
                    new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_UGC_ERROR_INVALID_GTU_PAGE").toSAX(contentHandler);
                    XMLUtils.endElement(contentHandler, "warning");
                }
                break;
            case "CONTENT":
                Optional<Content> gtuContent = _getGTUContent(zoneItem);
                if (gtuContent.isEmpty() || !_isValid(gtuContent.get()))
                {
                    XMLUtils.startElement(contentHandler, "warning");
                    new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_UGC_ERROR_INVALID_GTU_CONTENT").toSAX(contentHandler);
                    XMLUtils.endElement(contentHandler, "warning");
                }
                break;
            default:
                // Nothing to check
        }
        
        XMLUtils.endElement(contentHandler, "warnings");
    }
    
    private void _saxGTUIfNeeded(ZoneItem zoneItem, Request request) throws SAXException, MalformedURLException, IOException
    {
        String gtuMode = zoneItem.getServiceParameters().getValue("general-terms-of-use-mode", false, "NONE");
        if ("CONTENT".equals(gtuMode))
        {
            Optional<Content> gtuContent = _getGTUContent(zoneItem);
            if (gtuContent.isPresent())
            {
                Content content = gtuContent.get();
                XMLUtils.startElement(contentHandler, "gtu");
                
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", content.getId());
                attrs.addCDATAAttribute("name", content.getName());
                attrs.addCDATAAttribute("title", content.getTitle(null));
                attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
                
                XMLUtils.startElement(contentHandler, "content", attrs);
                
                String uri = _contentHelper.getContentHtmlViewUrl(content, "main");
                
                String currentSiteName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME);
                String currentSkinName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SKIN_ID);
                request.setAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR, currentSiteName);
                request.setAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR, currentSkinName);

                SitemapSource src = null;
                try
                {
                    src = (SitemapSource) _srcResolver.resolveURI(uri);
                    src.toSAX(new IgnoreRootHandler(contentHandler));
                }
                finally
                {
                    _srcResolver.release(src);
                    request.removeAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR);
                    request.removeAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR);
                    request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, currentSiteName);
                    request.setAttribute("siteName", currentSiteName);
                    request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, currentSkinName);
                }
                XMLUtils.endElement(contentHandler, "content");
                
                XMLUtils.endElement(contentHandler, "gtu");
            }
        }
    }
    
    private Optional<Page>  _getGTUPage(ZoneItem zoneItem)
    {
        String gtuPageId = zoneItem.getServiceParameters().getValue("general-terms-of-use-page");
        if (StringUtils.isNotEmpty(gtuPageId))
        {
            try
            {
                return Optional.of(_resolver.resolveById(gtuPageId));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The general terms of use page with id '" + gtuPageId + "' does not exist anymore", e);
            }
        }
        
        return Optional.empty();
    }
    
    private Optional<Content> _getGTUContent(ZoneItem zoneItem)
    {
        String gtuContentId = zoneItem.getServiceParameters().getValue("general-terms-of-use-content");
        if (StringUtils.isNotEmpty(gtuContentId))
        {
            try
            {
                return Optional.of(_resolver.resolveById(gtuContentId));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The general terms of use content with id '" + gtuContentId + "' does not exist anymore", e);
            }
        }
        
        return Optional.empty();
    }
    
    private boolean _isValid(Page page)
    {
        Skin skin = _skinManager.getSkin(page.getSite().getSkinId());
        
        Session liveSession = null;
        try
        {
            liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
            
            if (!_synchronizeComponent.isPageValid(page, skin) || !_synchronizeComponent.isHierarchyValid(page, liveSession))
            {
                return false;
            }
            return true;
        }
        catch (RepositoryException e)
        {
            throw new RuntimeException("Unable to check live workspace", e);
        }
        finally
        {
            if (liveSession != null)
            {
                liveSession.logout();
            }
        }
    }
    
    private boolean _isValid(Content content)
    {
        if (content instanceof VersionableAmetysObject vao)
        {
            return Arrays.asList(vao.getAllLabels()).contains(CmsConstants.LIVE_LABEL);
        }
        
        return false;
    }
}
