/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.ugc.page;

import java.util.Map;

import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} handling {@link UGCPage}.
 */
public class UGCPageFactory extends AbstractUGCPageFactory implements AmetysObjectFactory<UGCPage>
{
    /**
     * Create a UGC page.
     * @param root the root page.
     * @param syncContent the synchronized content
     * @param path the path
     * @return The <code>UGCPage</code> created
     */
    public UGCPage createUGCPage(Page root, Content syncContent, String path)
    {
        return new UGCPage(root, getConfiguration(root), getScheme(), this, syncContent, path);
    }
    
    public UGCPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // E.g: ugccontent://path?rootId=...&contentId=...
        String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "ugccontent://"), "?rootId=");
        String rootId = StringUtils.substringBefore(StringUtils.substringAfter(id, "?rootId="), "&contentId=");
        Page root = _resolver.resolveById(rootId);
        
        String contentId = StringUtils.substringAfter(id, "&contentId=");
        Content ugcContent = _resolver.resolveById(contentId);
        
        if ("_root".equals(path) && _ugcPageHandler.hasContentForRootPage(root, ugcContent))
        {
            return createUGCPage(root, ugcContent, path);
        }
        else
        {
            Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(root);
            if (transitionalPageName.containsKey(path))
            {
                Map<String, String> attributes = transitionalPageName.get(path);
                String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
                if (_ugcPageHandler.hasContentForTransitionalPage(root, metadataValue, ugcContent))
                {
                    return createUGCPage(root, ugcContent, path);
                }
            }
        }
        
        throw new UnknownAmetysObjectException("The UGC page with id '" + id  + "' does not match a existing UGC page for content with id " + contentId);
    }

    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // E.g: ugccontent://path?rootId=...&contentId=...
        String contentId = StringUtils.substringAfter(id, "&contentId=");
        
        if (!_resolver.hasAmetysObjectForId(contentId))
        {
            return false;
        }
        else
        {
            // Related content exists, check its page is the same
            UGCPage ugcPage = _ugcPageHandler.getUgcPage(contentId, null);
            return ugcPage != null ? id.equals(ugcPage.getId()) : false;
        }
    }

    public String getScheme()
    {
        return "ugccontent";
    }
}
