/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.AbstractConfigurableVirtualPage;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;


/**
 * Page representing a transitional page.
 */
public class UGCTransitionalPage extends AbstractConfigurableVirtualPage<UGCTransitionalPageFactory>
{
    private String _path;
    private String _title;
    private String _metadataValue;
    
    /**
     * Constructor.
     * @param root the user directory root page.
     * @param title the page's title.
     * @param metadataValue the metadata value
     * @param path the path
     * @param configuration The abstract virtual page's configuration
     * @param scheme The scheme
     * @param factory The UGC transitional page factory
     */
    public UGCTransitionalPage(Page root, VirtualPageConfiguration configuration, String scheme, UGCTransitionalPageFactory factory, String title, String metadataValue, String path)
    {
        super(root, configuration, scheme, factory);
       
        _title = title;
        _metadataValue = metadataValue;
        _path = path;
        _factory = factory;
    }
    
    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + 1;
    }

    @Override
    public Set<String> getReferers() throws AmetysRepositoryException
    {
        return null;
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _title;
    }

    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        
        AmetysObjectIterable<Content> contentsForTransitionalPage = _factory.getUgcPageHandler().getContentsForTransitionalPage(_root, _metadataValue);
        for (Content content : contentsForTransitionalPage)
        {
            children.add(_factory.getUGCPageFactory().createUGCPage(_root, content, _path));
        }
        
        return new CollectionIterable<>(children);
    }

    @Override
    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePage) throws AmetysRepositoryException
    {
        if (includeInvisiblePage)
        {
            return getChildrenPages();
        }
        else
        {
            List<Page> children = new ArrayList<>();
            return new CollectionIterable<>(children);
        }
    }
    
    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        return _root.getPathInSitemap() + "/" + _path;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        String name = path;
        AmetysObjectIterable<Content> contentsForTransitionalPage = _factory.getUgcPageHandler().getContentsForTransitionalPage(_root, _metadataValue);
        List<Content> contentFilters = contentsForTransitionalPage.stream().filter(c -> c.getName().equals(name)).collect(Collectors.toList());
        if (!contentFilters.isEmpty())
        {
            Content content = contentFilters.get(0);
            return (A) _factory.getUGCPageFactory().createUGCPage(_root, content, _path);
        }
        else
        {
            throw new UnknownAmetysObjectException("No ugc page for path " + path);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        AmetysObjectIterable<Content> contentsForTransitionalPage = _factory.getUgcPageHandler().getContentsForTransitionalPage(_root, _metadataValue);
        List<Content> contentFilters = contentsForTransitionalPage.stream().filter(c -> c.getName().equals(name)).collect(Collectors.toList());
        return !contentFilters.isEmpty();
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: ugctransitional://path?rootId=...
        return "ugctransitional://" + _path + "?rootId=" + _root.getId();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        return _path;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        return _root;
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return _root.getPath();
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    @Override
    public boolean isVisible() throws AmetysRepositoryException
    {
        return _factory.getUgcPageHandler().isClassificationPagesVisible(_root);
    }

    @Override
    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        return getChildrenPages().stream().collect(Collectors.toList()).get(index);
    }
}
