/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.ugc.page;

import java.util.Map;

import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} handling {@link UGCTransitionalPage}.
 */
public class UGCTransitionalPageFactory extends AbstractUGCPageFactory implements AmetysObjectFactory<UGCTransitionalPage>
{
    /**
     * Create a UGCTransitional page.
     *  @param root the user directory root page.
     * @param title the page's title.
     * @param metadataValue the metadata value
     * @param path the path
     * @return The <code>UGCPage</code> created
     */
    public UGCTransitionalPage createUGCTransitionalPage(Page root, String title, String metadataValue, String path)
    {
        return new UGCTransitionalPage(root, getConfiguration(root), getScheme(), this, title, metadataValue, path);
    }
    
    public UGCTransitionalPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // E.g: ugctransitional://path?rootId=...
        String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "ugctransitional://"), "?rootId=");
        
        String rootId = StringUtils.substringAfter(id, "?rootId=");
        Page root = _resolver.resolveById(rootId);
        
        Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(root);
        if (transitionalPageName.containsKey(path))
        {
            Map<String, String> attributes = transitionalPageName.get(path);
            String title = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_TITLE);
            String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
            return createUGCTransitionalPage(root, title, metadataValue, path);
        }
        else
        {
            throw new UnknownAmetysObjectException("No transitional page named " + path);
        }
    }

    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        try
        {
            // E.g: ugctransitional://path?rootId=...
            String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "ugctransitional://"), "?rootId=");
            
            String rootId = StringUtils.substringAfter(id, "?rootId=");
            Page root = _resolver.resolveById(rootId);
            
            Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(root);
            
            if (transitionalPageName.containsKey(path))
            {
                Map<String, String> attributes = transitionalPageName.get(path);
                String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
                
                AmetysObjectIterable<Content> contentsForTransitionalPage = _ugcPageHandler.getContentsForTransitionalPage(root, metadataValue);
                return contentsForTransitionalPage.getSize() != 0;
            }
            
            return false;
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
        
    }

    public String getScheme()
    {
        return "ugctransitional";
    }
}
