/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.indexing.IndexingObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.indexing.solr.SolrPageIndexer;
import org.ametys.web.repository.page.Page;

/**
 * {@link Observer} for observing UGC root is being modified or deleted
 */
public class SolrContentRootPageDeletedObserver extends AbstractLogEnabled implements IndexingObserver, Serviceable
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The Solr page indexer. */
    protected SolrPageIndexer _solrPageIndexer;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _solrPageIndexer = (SolrPageIndexer) manager.lookup(SolrPageIndexer.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_UGC_ROOT_DELETING)
                || event.getId().equals(ObservationConstants.EVENT_UGC_ROOT_UPDATING);
    }
    
    @Override
    public int getPriority()
    {
        return MAX_PRIORITY;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Page page = (Page) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_PAGE);
        
        // Unindex page recursively before updating page properties and before live synchronization
        _solrPageIndexer.unindexPage(page.getId(), true, true);
    }
}
