/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.rights;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.frontedition.AmetysFrontEditionHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.rights.PageAccessController;

/**
 * This controller will grant the right "Front_Edition_Access_Right" on the {@link UserPage} to any connected user
 */
public class EditionFOAccessController implements AccessController, Serviceable, Contextualizable, PluginAware
{
    /** the user generated page handler */
    protected UserDirectoryPageHandler _udPageHandler;
    /** The site manager */
    protected SiteManager _siteManager;
    private String _id;
    private Context _context;

    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _udPageHandler = (UserDirectoryPageHandler) manager.lookup(UserDirectoryPageHandler.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID.equals(rightId) && object instanceof UserPage)
        {
            return AccessResult.USER_ALLOWED;
        }
        else
        {
            return AccessResult.UNKNOWN;
        }
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        AccessResult accessResult = getPermission(user, userGroups, AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID, object);
        if (accessResult == AccessResult.UNKNOWN)
        {
            return Collections.EMPTY_MAP;
        }
        else
        {
            Map<String, AccessResult> permissions = new HashMap<>();
            permissions.put(AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID, accessResult);
            return permissions;
        }
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        if (AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID.equals(rightId) && object instanceof UserPage)
        {
            return AccessResult.ANY_CONNECTED_ALLOWED;
        }
        else
        {
            return AccessResult.UNKNOWN;
        }
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        // We do not want that this accesscontroller give access to the backoffice (even if #isSupported would not match in this case)
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        // We do not want that this accesscontroller give access to the backoffice (even if #isSupported would not match in this case)
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        // We do not want that this accesscontroller give access to the backoffice (even if #isSupported would not match in this case)
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean isSupported(Object object)
    {
        return object instanceof UserPage;
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case USER_ALLOWED:
            case ANY_CONNECTED_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(
                        getId(),
                        permission,
                        new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_FO_ACCESS_CONTROLLER_" + permission.name() + "_EXPLANATION",
                                Map.of("page", getObjectLabel(object)))
                    );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), permission);
        }
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups)
    {
        Map<ExplanationObject, Map<Permission, AccessExplanation>> result = new HashMap<>();
        
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        Site site = _siteManager.getSite(siteName);
        
        try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
        {
            for (Sitemap sitemap : sitemaps)
            {
                Set<Page> rootPages = _udPageHandler.getUserDirectoryRootPages(siteName, sitemap.getSitemapName());
                if (!rootPages.isEmpty())
                {
                    ExplanationObject context = new ExplanationObject(
                            "all_user_pages",
                            new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_FO_ACCESS_CONTROLLER_ALL_USER_PAGE"),
                            getObjectCategory(null),
                            10
                        );
                    result.put(context, Map.of(new Permission(PermissionType.RIGHT, AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID), getStandardAccessExplanation(AccessResult.USER_ALLOWED, rootPages.stream().findAny().get())));
                    return result;
                }
            }
        }
        
        return result;
    }

    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Page page)
        {
            return new I18nizableText(page.getTitle());
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return PageAccessController.PAGE_CONTEXT_CATEGORY;
    }

    public String getId()
    {
        return _id;
    }
}
