/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.transformation.xslt;

import java.io.InputStream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.File;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;

/**
 * Implementation of {@link ProfileImageResolverHelper} to get the profile image from user content if exists
 */
public class ProfileImageResolverHelper extends org.ametys.web.transformation.xslt.ProfileImageResolverHelper
{
    /** Attribute path for user's image */
    public static final String USER_CONTENT_IMAGE_PATH = "illustration/image";
    
    /** The user directory helper */
    protected static UserDirectoryHelper _userDirectoryHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
    }
    
    @Override
    public String getProfileImageUri(String login, String populationId, int size)
    {
        return "/_plugins/user-directory/user/" + populationId + "/" + login + "/image_" + size;
    }
    
    @Override
    public InputStream getProfileImage(String login, String populationId, int cropSize, String lang)
    {
        UserIdentity user = new UserIdentity(login, populationId);
        
        File file = getProfileImageFileFromUserContentIfExists(user, lang);
        if (file != null)
        {
            return file.getInputStream();
        }
        
        return super.getProfileImage(login, populationId, cropSize, lang);
    }
    
    /**
     * Get the image uri from user content if exists
     * @param user the user identity
     * @param lang the content language
     * @return the file image or null if not found
     */
    public File getProfileImageFileFromUserContentIfExists(UserIdentity user, String lang)
    {
        Content userContent = _userDirectoryHelper.getUserContent(user, lang);
        if (userContent != null && userContent.hasValue(USER_CONTENT_IMAGE_PATH))
        {
            return userContent.getValue(USER_CONTENT_IMAGE_PATH);
        }
        
        return null;
    }
}
