/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.clientsideelement;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.value.StringValue;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.observation.ObservationConstants;
import org.ametys.plugins.userdirectory.page.VirtualUserDirectoryPageFactory;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.clientsideelement.AbstractPageClientSideElement;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.rights.PageRightAssignmentContext;

/**
 * Client side element for a controller wich set/remove the root page of a user directory.
 */
public class SetUserDirectoryRootClientSideElement extends AbstractPageClientSideElement
{
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The extension point for content types */
    protected ContentTypeExtensionPoint _contentTypeEP;
    /** The User Directory page handler */
    protected UserDirectoryPageHandler _userDirectoryPageHandler;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _userDirectoryPageHandler = (UserDirectoryPageHandler) smanager.lookup(UserDirectoryPageHandler.ROLE);
    }
    
    /**
     * Gets the status of the given page
     * @param pageId The page id
     * @return the status of the given page
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getStatus(String pageId)
    {
        // Assume that no read access is checked (required to update client side element status)
        
        Map<String, Object> result = new HashMap<>();
        
        Map<String, Object> parameters = this._script.getParameters();
        
        Page page = _resolver.resolveById(pageId);

        if (page instanceof JCRAmetysObject)
        {
            if (_isUserDirectoryRootPage((JCRAmetysObject) page))
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("user-directory-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("user-directory-page-title", msg);
                
                ed = (I18nizableText) parameters.get("remove-user-directory-page-description");
                msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("remove-user-directory-page-title", msg);
                
                String contentTypeId = page.getValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME, StringUtils.EMPTY);
                
                if (StringUtils.isNotEmpty(contentTypeId))
                {
                    I18nizableText contentTypeText = _contentTypeEP.hasExtension(contentTypeId) ? _contentTypeEP.getExtension(contentTypeId).getLabel() : new I18nizableText(contentTypeId);
                    
                    Map<String, I18nizableTextParameter> contentTypeI18nParameters = new HashMap<>();
                    contentTypeI18nParameters.put("0", contentTypeText);
                    
                    ed = (I18nizableText) parameters.get("contenttype-user-directory-page-description");
                    msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), contentTypeI18nParameters);
                    result.put("contenttype-user-directory-page-description", msg);
                }
                
                result.put("user-directory-page-id", new I18nizableText(page.getId()));
                
                I18nizableText defaultDirectoryDeep = (I18nizableText) parameters.get("default-depth");
                long depthData = page.getValue(UserDirectoryPageHandler.DEPTH_DATA_NAME, -1L);
                I18nizableText depth = depthData == -1 ? defaultDirectoryDeep : new I18nizableText(String.valueOf(depthData));
                result.put("depth", depth);
            }
            else
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("add-user-directory-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                
                result.put("add-user-directory-page-id", new I18nizableText(page.getId()));
                result.put("add-user-directory-page-title", msg);
            }
        }
        else
        {
            List<String> noJcrI18nParameters = new ArrayList<>();
            noJcrI18nParameters.add(page.getTitle());

            I18nizableText ed = (I18nizableText) parameters.get("no-jcr-page-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), noJcrI18nParameters);
            
            result.put("no-jcr-page-id", new I18nizableText(page.getId()));
            result.put("no-jcr-page-title", msg);
        }
        
        return result;
    }
    
    private boolean _isUserDirectoryRootPage (JCRAmetysObject jcrPage)
    {
        try
        {
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues());
                
                return values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .anyMatch(v -> VirtualUserDirectoryPageFactory.class.getName().equals(v));
            }
            else
            {
                return false;
            }
        }
        catch (RepositoryException e)
        {
            return false;
        }
    }
    
    /**
     * Gets the content types which can build a user directory
     * @param pageId The id of the page being edited
     * @return the content types which can build a user directory
     */
    @Callable (rights = "User_Directory_Right_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public List<Map<String, Object>> getSupportedContentTypes(String pageId)
    {
        List<Map<String, Object>> result = new ArrayList<>();
        Page page = _resolver.resolveById(pageId);
        
        for (String contentTypeId : _contentTypeEP.getSubTypes(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE))
        {
            ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
            Page userDirectoryRootPage = _userDirectoryPageHandler.getUserDirectoryRootPage(page.getSiteName(), page.getSitemapName(), contentTypeId);
            if (!contentType.isAbstract() && (userDirectoryRootPage == null || userDirectoryRootPage.equals(page)))
            {
                // The content type is not already a root of a user directory or is the root of the currently edited page
                Map<String, Object> entry = new HashMap<>();
                entry.put("value", contentType.getId());
                entry.put("text", contentType.getLabel());
                result.add(entry);
            }
        }
        
        return result;
    }
    
    /**
     * Sets the given page as the root of a user directory
     * @param pageId The id of the page
     * @param contentType The id of the content type
     * @param viewName The view name for users rendering
     * @param attribute The classification attribute
     * @param depth The depth of the tree structure
     * @return A result map
     * @throws RepositoryException if a repository error occurred
     */
    @Callable (rights = "User_Directory_Right_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> setUserDirectoryRoot(String pageId, String contentType, String viewName, String attribute, int depth) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        if (!_contentTypeEP.isDescendant(contentType, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE))
        {
            result.put("error", "invalid-content-type");
            return result;
        }
        
        Page page = _resolver.resolveById(pageId);
        String oldContentType = page.getValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME, StringUtils.EMPTY);
        String oldAttribute = page.getValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, StringUtils.EMPTY);
        String oldViewName = page.getValue(UserDirectoryPageHandler.USER_VIEW_NAME, StringUtils.EMPTY);
        long oldDepth = page.getValue(UserDirectoryPageHandler.DEPTH_DATA_NAME, -1L);
        
        // Do nothing if page attribute are the same
        if (!oldContentType.equals(contentType) || !oldAttribute.equals(attribute) || oldDepth != depth || !oldViewName.equals(viewName))
        {
            Set<Page> currentUserDirectoryPages = _userDirectoryPageHandler.getUserDirectoryRootPages(page.getSiteName(), page.getSitemapName());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            
            if (currentUserDirectoryPages.contains(page))
            {
                eventParams.put(ObservationConstants.ARGS_USER_CONTENT_VIEW_UPDATED, !oldViewName.equals(viewName));
                
                // Unindex pages for all workspaces before the properties changed 
                _observationManager.notify(new Event(ObservationConstants.EVENT_USER_DIRECTORY_ROOT_UPDATING, _currentUserProvider.getUser(), eventParams));
                
                _updateUserDirectoryRootProperty(page, contentType, viewName, attribute, depth);
            }
            else
            {
                _addUserDirectoryRootProperty(page, contentType, viewName, attribute, depth);
            }
            
            _userDirectoryPageHandler.clearCache(page);
            
            // Live synchronization
            _notifyPageUpdated(page);
            
            // Indexation
            _observationManager.notify(new Event(ObservationConstants.EVENT_USER_DIRECTORY_ROOT_UPDATED, _currentUserProvider.getUser(), eventParams));
        }
        
        return result;
    }
    
    /**
     * Remove the user directory root status to the given page
     * @param pageId The id of the page
     * @return A result map
     * @throws RepositoryException if a repository error occured
     */
    @Callable (rights = "User_Directory_Right_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> removeUserDirectoryRoot(String pageId) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (page instanceof JCRAmetysObject)
        {
            if (!_isUserDirectoryRootPage((JCRAmetysObject) page))
            {
                result.put("error", "no-root");
                return result;
            }
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            
            // Unindex pages for all workspaces before the properties were removed 
            _observationManager.notify(new Event(ObservationConstants.EVENT_USER_DIRECTORY_ROOT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            _userDirectoryPageHandler.clearCache(page);
            _removeUserDirectoryRootProperty(page);
            
            _notifyPageUpdated(page);
            
            // After live synchronization
            _observationManager.notify(new Event(ObservationConstants.EVENT_USER_DIRECTORY_ROOT_DELETED, _currentUserProvider.getUser(), eventParams));
        }
        else
        {
            result.put("error", "no-root");
        }
        return result;
    }
    
    /**
     * Gets information about user directory root status on the given.
     * @param pageId The id of the page
     * @return information about user directory root status on the given.
     */
    @Callable (rights = "User_Directory_Right_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getRootPageInfo(String pageId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        Set<Page> currentUserDirectoryPages = _userDirectoryPageHandler.getUserDirectoryRootPages(page.getSiteName(), page.getSitemapName());
        
        if (currentUserDirectoryPages.contains(page))
        {
            result.put("isRoot", true);
            result.put("contentType", page.getValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME));
            result.put("viewName", page.getValue(UserDirectoryPageHandler.USER_VIEW_NAME, StringUtils.EMPTY));
            result.put("metadata", page.getValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME));
            result.put("depth", page.getValue(UserDirectoryPageHandler.DEPTH_DATA_NAME));
        }
        else
        {
            result.put("isRoot", false);
        }
        
        return result;
    }
    
    private void _addUserDirectoryRootProperty(Page page, String contentType, String viewName, String attribute, int depth) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            List<Value> values = new ArrayList<>();
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                values.addAll(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
            }
            
            StringValue virtualUserDirectoryPageFactoryClassName = new StringValue(VirtualUserDirectoryPageFactory.class.getName());
            if (!values.contains(virtualUserDirectoryPageFactoryClassName))
            {
                values.add(virtualUserDirectoryPageFactoryClassName);
            }
            
            node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));

            // Set the user directory root property
            if (page instanceof ModifiablePage)
            {
                ModifiablePage modifiablePage = (ModifiablePage) page;
                modifiablePage.setValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
                modifiablePage.setValue(UserDirectoryPageHandler.USER_VIEW_NAME, viewName);
                modifiablePage.setValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, attribute);
                modifiablePage.setValue(UserDirectoryPageHandler.DEPTH_DATA_NAME, depth);
            }
            
            jcrPage.saveChanges();
        }
    }
    
    private void _updateUserDirectoryRootProperty(Page page, String contentType, String viewName, String attribute, int depth)
    {
        if (page instanceof ModifiablePage)
        {
            ModifiablePage modifiablePage = (ModifiablePage) page;
            
            // Set the user directory root property
            modifiablePage.setValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
            modifiablePage.setValue(UserDirectoryPageHandler.USER_VIEW_NAME, viewName);
            modifiablePage.setValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, attribute);
            modifiablePage.setValue(UserDirectoryPageHandler.DEPTH_DATA_NAME, depth);
            
            modifiablePage.saveChanges();
        }
    }
    
    private void _removeUserDirectoryRootProperty(Page page) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = new ArrayList<>(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
                int index = values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .collect(Collectors.toList())
                        .indexOf(VirtualUserDirectoryPageFactory.class.getName());
                if (index != -1)
                {
                    values.remove(index);
                    node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));
                }

                // Remove the user directory root property
                if (page instanceof ModifiablePage)
                {
                    ModifiablePage modifiablePage = (ModifiablePage) page;
                    modifiablePage.removeValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME);
                    modifiablePage.removeValue(UserDirectoryPageHandler.USER_VIEW_NAME);
                    modifiablePage.removeValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME);
                    modifiablePage.removeValue(UserDirectoryPageHandler.DEPTH_DATA_NAME);
                }
                
                jcrPage.saveChanges();
            }
        }
    }
    
    private void _notifyPageUpdated(Page page)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
        _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_CHANGED, _currentUserProvider.getUser(), eventParams));
    }
}
