/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.userdirectory.page;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} handling {@link UserPage}.
 */
public class UserPageFactory extends AbstractUserDirectoryPageFactory implements AmetysObjectFactory<UserPage>
{
    private SynchronizableContentsCollectionHelper _sccHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);

        _sccHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
    }

    /**
     * Create a User page.
     * @param root the root page.
     * @param syncContent the synchronized content
     * @param path the path
     * @return The <code>UserPager</code> created
     */
    public UserPage createUserPage(Page root, Content syncContent, String path)
    {
        return new UserPage(root, getConfiguration(), getScheme(), this, syncContent, path);
    }
    
    public UserPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // E.g: uduser://path?rootId=...&contentId=...
        String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "uduser://"), "?rootId=");
        String rootId = StringUtils.substringBefore(StringUtils.substringAfter(id, "?rootId="), "&contentId=");
        Page root = _resolver.resolveById(rootId);
        
        String contentId = StringUtils.substringAfter(id, "&contentId=");
        Map<String, String> userPagesContent = _userDirectoryPageHandler.getUserPagesContent(root, path);
        if (userPagesContent.values().contains(contentId))
        {
            Content syncContent = _resolver.resolveById(contentId);
            return createUserPage(root, syncContent, path);
        }
        else
        {
            throw new UnknownAmetysObjectException("No user content for id " + contentId);
        }
    }

    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // Id is like uduser://path?rootId=...&contentId=...
        String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "uduser://"), "?rootId=");
        String rootId = StringUtils.substringBefore(StringUtils.substringAfter(id, "?rootId="), "&contentId=");
        if (!_resolver.hasAmetysObjectForId(rootId))
        {
            return false;
        }
        
        Page root = _resolver.resolveById(rootId);
        
        String contentId = StringUtils.substringAfter(id, "&contentId=");
        Map<String, String> userPagesContent = _userDirectoryPageHandler.getUserPagesContent(root, path);
        return userPagesContent.values().contains(contentId);
    }

    public String getScheme()
    {
        return "uduser";
    }
    
    /**
     * Get the synchronizable contents collection helper
     * @return The <code>SynchronizableContentsCollectionHelper</code>
     */
    public SynchronizableContentsCollectionHelper getSccHelper()
    {
        return _sccHelper;
    }
}
