/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} for handling "virtual" user directory page
 */
public class VirtualUserDirectoryPageFactory extends AbstractUserDirectoryPageFactory implements VirtualAmetysObjectFactory<Page>
{
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _userDirectoryPageHandler = (UserDirectoryPageHandler) manager.lookup(UserDirectoryPageHandler.ROLE);
    }

    @Override
    public Page getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public String getScheme()
    {
        return "udroot";
    }

    @Override
    public AmetysObjectIterable<Page> getChildren(JCRAmetysObject parent)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the user directory virtual pages should be a page.");
        }
        
        List<Page> children = new ArrayList<>();
        Page rootPage = (Page) parent;
        
        int depth = _userDirectoryPageHandler.getDepth(rootPage);
        if (depth > 0)
        {
            SortedSet<String> transitionalPagesName = _userDirectoryPageHandler.getTransitionalPagesName(rootPage, "_root");
            for (String pageName : transitionalPagesName)
            {
                String pathName = _userDirectoryPageHandler.getPathName(pageName);
                children.add(getTransitionalPageFactory().createTransitionalPage(rootPage, pageName, pathName));
            }
        }
        else
        {
            Map<String, String> userPagesContent = _userDirectoryPageHandler.getUserPagesContent(rootPage, "_root");
            for (String contentId : userPagesContent.values())
            {
                if (_resolver.hasAmetysObjectForId(contentId))
                {
                    Content content = _resolver.resolveById(contentId);
                    children.add(getUserPageFactory().createUserPage(rootPage, content, "_root"));
                }
                else
                {
                    getLogger().info("Content '{}' is stil referenced, but it does not exists anymore");
                }
            }
        }
        
        return new CollectionIterable<>(children);
    }

    @Override
    public Page getChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the user directory virtual pages should be a page.");
        }
        
        Page rootPage = (Page) parent;
        int intialDepth = _userDirectoryPageHandler.getDepth(rootPage);
        
        if (intialDepth > 0)
        {
            SortedSet<String> transitionalPagesName = _userDirectoryPageHandler.getTransitionalPagesName(rootPage, "_root");
            String name = _userDirectoryPageHandler.getName(childName);
            if (transitionalPagesName.contains(name))
            {
                return getTransitionalPageFactory().createTransitionalPage(rootPage, name, childName);
            }
            else
            {
                throw new UnknownAmetysObjectException("No transitional page named " + childName);
            }
        }
        else
        {
            Map<String, String> userPagesContent = _userDirectoryPageHandler.getUserPagesContent(rootPage, "_root");
            if (userPagesContent.containsKey(childName))
            {
                Content content = _resolver.resolveById(userPagesContent.get(childName));
                return getUserPageFactory().createUserPage(rootPage, content, "_root");
            }
            else
            {
                throw new UnknownAmetysObjectException("No user content named " + childName);
            }
        }
    }

    @Override
    public boolean hasChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the user directory virtual pages should be a page.");
        }
        
        Page rootPage = (Page) parent;
        int intialDepth = _userDirectoryPageHandler.getDepth(rootPage);
        
        if (intialDepth > 0)
        {
            SortedSet<String> transitionalPagesName = _userDirectoryPageHandler.getTransitionalPagesName(rootPage, "_root");
            String name = _userDirectoryPageHandler.getName(childName);
            return transitionalPagesName.contains(name);
        }
        else
        {
            Map<String, String> userPagesContent = _userDirectoryPageHandler.getUserPagesContent(rootPage, childName);
            return userPagesContent.containsKey(childName);
        }
    }
}
