/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.Logger;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.clientsideelement.content.SmartContentClientSideElementHelper;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;

/**
 * Delete UD content component
 */
public abstract class AbstractDeleteUDContentComponent implements Component, Serviceable, Configurable
{
    private static final int _REMOVE_REFERENCE_DEFAULT_ACTION_ID = 200;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** Helper for smart content client elements */
    protected SmartContentClientSideElementHelper _smartHelper;
    
    /** The action id to call when references are removed */
    protected int _removeReferenceActionId;

    private SolrIndexHelper _solrIndexHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _smartHelper = (SmartContentClientSideElementHelper) smanager.lookup(SmartContentClientSideElementHelper.ROLE);
        _solrIndexHelper = (SolrIndexHelper) smanager.lookup(SolrIndexHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration conf = configuration.getChild("removeReferenceActionId");
        _removeReferenceActionId = conf.getValueAsInteger(_REMOVE_REFERENCE_DEFAULT_ACTION_ID);
    }
    
    /**
     * Delete contents and logs results
     * @param contentsToRemove the list of contents to remove
     * @param parameters the additional parameters
     * @param rights the map of rights id with its prefix
     * @param logger The logger
     * @return the number of deleted contents
     */
    @SuppressWarnings("unchecked")
    public int deleteContentsWithLog(List<Content> contentsToRemove, Map<String, Object> parameters, Map<String, String> rights, Logger logger)
    {
        int nbDeletedContents = 0;
        
        List<String> contentIds = contentsToRemove.stream()
                .map(Content::getId)
                .collect(Collectors.toList());
        
        logger.info("Trying to delete contents. This can take a while...");
        Map<String, Object> deleteResults = deleteContents(contentIds, parameters, rights, logger);
        logger.info("Contents deleting process ended.");
        
        for (String contentId : contentIds)
        {
            Map<String, Object> result = (Map<String, Object>) deleteResults.get(contentId);
            if (result != null) // if the result is null, the content was already deleted because it's a child of a previous deleted content
            {
                List<String> deletedContents = (List<String>) result.get("deleted-contents");
                nbDeletedContents += deletedContents.size();
                
                List<Content> referencedContents = (List<Content>) result.get("referenced-contents");
                if (referencedContents.size() > 0)
                {
                    logger.info("The following contents cannot be deleted because they are referenced: {}", referencedContents.stream().map(c -> c.getId()).collect(Collectors.toList()));
                }
                
                List<Content> lockedContents = (List<Content>) result.get("locked-contents");
                if (lockedContents.size() > 0)
                {
                    logger.info("The following contents cannot be deleted because they are locked: {}", lockedContents.stream().map(c -> c.getId()).collect(Collectors.toList()));
                }
                
                List<Content> undeletedContents = (List<Content>) result.get("undeleted-contents");
                if (undeletedContents.size() > 0)
                {
                    logger.info("{} contents were not deleted. See previous logs for more information.", undeletedContents.size());
                }
            }
        }
        
        return nbDeletedContents;
    }
    
    /**
     * Delete contents
     * @param contentsId The ids of contents to delete
     * @param parameters the additional parameters
     * @param rights the map of rights id with its prefix
     * @param logger The logger
     * @return the deleted and undeleted contents
     */
    public Map<String, Object> deleteContents(List<String> contentsId, Map<String, Object> parameters, Map<String, String> rights, Logger logger)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<String> alreadyDeletedContentIds = new ArrayList<>();
        for (String contentId : contentsId)
        {
            if (!alreadyDeletedContentIds.contains(contentId))
            {
                Content content = _resolver.resolveById(contentId);
                
                Map<String, Object> result = new HashMap<>();
                result.put("deleted-contents", new ArrayList<>());
                result.put("undeleted-contents", new ArrayList<>());
                result.put("referenced-contents", new ArrayList<>());
                result.put("unauthorized-contents", new ArrayList<>());
                result.put("locked-contents", new ArrayList<>());
                result.put("initial-content", content.getId());
                results.put(contentId, result);
                
                boolean referenced = isContentReferenced(content, logger);
                if (referenced || !_checkBeforeDeletion(content, rights, result, logger))
                {
                    if (referenced)
                    {
                        // Indicate that the content is referenced.
                        @SuppressWarnings("unchecked")
                        List<Content> referencedContents = (List<Content>) result.get("referenced-contents");
                        referencedContents.add(content);
                    }
                    result.put("check-before-deletion-failed", true);
                }
                else
                {
                    // Process deletion
                    _deleteContent(content, parameters, rights, result, logger);

                    @SuppressWarnings("unchecked")
                    List<String> deletedContents = (List<String>) result.get("deleted-contents");
                    if (deletedContents != null)
                    {
                        alreadyDeletedContentIds.addAll(deletedContents);
                        
                    }
                }
            }
            else
            {
                logger.info("Content with id '{}' has been already deleted during its parent deletion", contentId);
            }
        }

        return results;
    }

    /**
     * Delete one content
     * @param content the content to delete
     * @param parameters the additional parameters
     * @param rights the map of rights id with its prefix
     * @param results the results map
     * @param logger The logger
     */
    protected void _deleteContent(Content content, Map<String, Object> parameters, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        // 1 - First remove relations
        boolean success = _removeRelations(content, parameters, logger);
        
        // 2 - If succeed, process to deletion
        if (success)
        {
            _processContentDeletion(content, parameters, rights, results, logger);
        }
        else
        {
            @SuppressWarnings("unchecked")
            List<Content> undeletedContents = (List<Content>) results.get("undeleted-contents");
            undeletedContents.add(content);
            
            logger.warn("Can not delete content {} ('{}') : at least one relation to contents could not be removed", content.getTitle(), content.getId());
        }
    }

    /**
     * Delete one content
     * @param content the content to delete
     * @param parameters the additional parameters
     * @param rights the map of rights id with its prefix
     * @param results the results map
     * @param logger The logger
     */
    @SuppressWarnings("unchecked")
    protected void _processContentDeletion(Content content, Map<String, Object> parameters, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        Set<String> toDelete = _getContentIdsToDelete(content, parameters, rights, results, logger);
        
        List<Content> referencedContents = (List<Content>) results.get("referenced-contents");
        List<Content> lockedContents = (List<Content>) results.get("locked-contents");
        List<Content> unauthorizedContents = (List<Content>) results.get("unauthorized-contents");
        
        if (referencedContents.size() == 0 && lockedContents.size() == 0 && unauthorizedContents.size() == 0)
        {
            _finalizeDeleteContents(toDelete, content.getParent(), results, logger);
        }
    }
    
    /**
     * Finalize the deletion of contents. Call observers and remove contents
     * @param contentIdsToDelete the list of content id to delete
     * @param parent the jcr parent for saving changes
     * @param results the results map
     * @param logger The logger
     */
    protected void _finalizeDeleteContents(Set<String> contentIdsToDelete, ModifiableAmetysObject parent, Map<String, Object> results, Logger logger)
    {
        @SuppressWarnings("unchecked")
        List<Content> unauthorizedContents = (List<Content>) results.get("unauthorized-contents");
        @SuppressWarnings("unchecked")
        List<Content> lockedContents = (List<Content>) results.get("locked-contents");
        
        if (!unauthorizedContents.isEmpty() || !lockedContents.isEmpty())
        {
            //Do Nothing
            return;
        }
        
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(new String[] {ObservationConstants.EVENT_CONTENT_DELETED});
            
            Map<String, Map<String, Object>> eventParams = new HashMap<>();
            for (String id : contentIdsToDelete)
            {
                Content content = _resolver.resolveById(id);
                Map<String, Object> eventParam = _getEventParametersForDeletion(content);
                eventParams.put(id, eventParam);
                
                _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETING, _currentUserProvider.getUser(), eventParam));
            }
            
            for (String id : contentIdsToDelete)
            {
                Content content = _resolver.resolveById(id);
                
                // Remove the content.
                LockableAmetysObject lockedContent = (LockableAmetysObject) content;
                if (lockedContent.isLocked())
                {
                    lockedContent.unlock();
                }
                
                ((RemovableAmetysObject) content).remove();
            }
            
            parent.saveChanges();
            
            for (String id : contentIdsToDelete)
            {
                _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETED, _currentUserProvider.getUser(), eventParams.get(id)));
                
                @SuppressWarnings("unchecked")
                List<String> deletedContents = (List<String>) results.get("deleted-contents");
                deletedContents.add(id);
            }
        }
        finally 
        {
            _solrIndexHelper.restartSolrCommitForEvents(new String[] {ObservationConstants.EVENT_CONTENT_DELETED});
        }
    }
    
    /**
     * True if we can delete the content (check if removable, rights and if locked)
     * @param content the content
     * @param rights the map of rights id with its prefix
     * @param results the results map
     * @return true if we can delete the content
     */
    protected boolean _canDeleteContent(Content content, Map<String, String> rights, Map<String, Object> results)
    {
        if (!(content instanceof RemovableAmetysObject))
        {
            throw new IllegalArgumentException("The content [" + content.getId() + "] is not a RemovableAmetysObject, it can't be deleted.");
        }
        
        if (!_hasRight(content, rights))
        {
            // User has no sufficient right
            @SuppressWarnings("unchecked")
            List<Content> norightContents = (List<Content>) results.get("unauthorized-contents");
            norightContents.add(content);
            
            return false;
        }
        else if (_isLocked(content))
        {
            @SuppressWarnings("unchecked")
            List<Content> lockedContents = (List<Content>) results.get("locked-contents");
            lockedContents.add(content);
            
            return false;
        }
        
        return true;
    }
    
    /**
     * Get parameters for content deleted {@link Event}
     * @param content the removed content
     * @return the event's parameters
     */
    protected Map<String, Object> _getEventParametersForDeletion (Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, content.getName());
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        return eventParams;
    }
    
    /**
     * Determines if the content is locked
     * @param content the content
     * @return true if the content is locked
     */
    protected boolean _isLocked(Content content)
    {
        return _smartHelper.isLocked(content);
    }
    
    /**
     * Determines if the user has sufficient right for the given content
     * @param content the content
     * @param rights the map of rights id with its prefix
     * @return true if user has sufficient right
     */
    protected boolean _hasRight(Content content, Map<String, String> rights)
    {
        if (rights.isEmpty())
        {
            return true;
        }
        
        return _smartHelper.hasRight(rights, content);
    }
    
    /**
     * True if the content is referenced
     * @param content the content
     * @param logger The logger
     * @return true if the content is referenced
     */
    public abstract boolean isContentReferenced(Content content, Logger logger);

    /**
     * Check that deletion can be performed without blocking errors
     * @param content The initial content to delete
     * @param rights the map of rights id with its prefix
     * @param results The results
     * @param logger The logger
     * @return true if the deletion can be performed
     */
    protected abstract boolean _checkBeforeDeletion(Content content, Map<String, String> rights, Map<String, Object> results, Logger logger);
    
    /**
     * Remove relations
     * @param content the content
     * @param parameters the additional parameters
     * @param logger The logger
     * @return <code>true</code> if all relations have been removed
     */
    protected abstract boolean _removeRelations(Content content, Map<String, Object> parameters, Logger logger);
    
    /**
     * Get the id of children to be deleted. 
     * All children shared with other contents which are not part of deletion, will be not deleted.
     * @param content The content to delete
     * @param parameters the additional parameters
     * @param rights the map of rights id with its prefix
     * @param results The results
     * @param logger The logger
     * @return The id of contents to be deleted
     */
    protected abstract Set<String> _getContentIdsToDelete (Content content, Map<String, Object> parameters, Map<String, String> rights, Map<String, Object> results, Logger logger);
    
}
