/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Delete user component
 */
public class DeleteUserComponent extends AbstractDeleteUDContentComponent
{
    /** The avalon role. */
    public static final String ROLE = DeleteUserComponent.class.getName();
   
    /** The user directory helper */
    protected UserDirectoryHelper _udHelper;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _udHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    public boolean isContentReferenced(Content content, Logger logger)
    {
        return _contentHelper.hasReferencingContents(content, Collections.singletonList(UserDirectoryHelper.ORGUNIT_CONTENT_TYPE), true);
    }
    
    @Override
    protected boolean _checkBeforeDeletion(Content content, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        // Check right and lock on content it self
        boolean allRight = _canDeleteContent(content, rights, results);
        
        // Check lock on orgunits contents
        allRight = _checkOrgUnitsBeforeDeletion(content, results) && allRight;
        
        return allRight;
    }
    
    /**
     * True if the parent content is not locked
     * @param content the parent content
     * @param results the results map
     * @return true if the parent content is not locked
     */
    protected boolean _checkOrgUnitsBeforeDeletion(Content content,  Map<String, Object> results)
    {
        boolean allRight = true;
        
        // Check if parents are not locked
        List<Content> orgUnits = _udHelper.getOrgUnits(content);
        for (Content orgUnit : orgUnits)
        {
            if (_isLocked(orgUnit))
            {
                @SuppressWarnings("unchecked")
                List<Content> lockedContents = (List<Content>) results.get("locked-contents");
                lockedContents.add(orgUnit);
                
                allRight = false;
            }
        }
        
        return allRight;
    }

    @Override
    protected boolean _removeRelations(Content content, Map<String, Object> parameters, Logger logger)
    {
        boolean success = true;
        
        // Delete relation to orgunits
        List<Pair<String, Content>> incomingReferences = _contentHelper.getReferencingContents(content);
        for (Pair<String, Content> reference : incomingReferences)
        {
            String valuePath = reference.getLeft();
            Content orgUnit = reference.getRight();
            
            try
            {
                I18nizableText commentText = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_WORKFLOW_ACTION_REMOVE_USER_REFERENCE_MSG");
                String comment = _i18nUtils.translate(commentText, orgUnit.getLanguage());
                
                _removeRelation((WorkflowAwareContent) orgUnit, valuePath, comment);
            }
            catch (WorkflowException | InvalidActionException e)
            {
                logger.error("Unable to remove relation to content \"{}\" ({}) for referencing content \"{}\" ({}) ", content.getTitle(), content.getId(), orgUnit.getTitle(), orgUnit.getId(), e);
                success = false;
            }
        }
        
        return success;
    }
    
    private void _removeRelation(WorkflowAwareContent orgUnit, String valuePath, String comment) throws WorkflowException
    {
        // The valuePath is like users[1]/user
        String entryPath = StringUtils.substringBeforeLast(valuePath, ModelItem.ITEM_PATH_SEPARATOR);
        
        int position = DataHolderHelper.getRepeaterNameAndEntryPosition(entryPath).getRight();
        
        SynchronizableRepeater repeater = SynchronizableRepeater.appendOrRemove(List.of(), Set.of(position));
        Map<String, Object> values = Map.of("users", repeater);
        
        Map<String, Object> inputs = new HashMap<>();
        if (StringUtils.isNotEmpty(comment))
        {
            inputs.put("comment", comment);
        }
        
        Map<String, Object> parameters = new HashMap<>();
        
        parameters.put(EditContentFunction.VALUES_KEY, values);
        parameters.put(EditContentFunction.QUIT, true);
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, parameters);
    
        _contentWorkflowHelper.doAction(orgUnit, _removeReferenceActionId, inputs);
    }

    @Override
    protected Set<String> _getContentIdsToDelete(Content content, Map<String, Object> parameters, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        return Collections.singleton(content.getId());
    }
}
