/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.userdirectory.page;

import java.util.SortedSet;

import org.apache.commons.lang.StringUtils;

import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} handling {@link TransitionalPage}.
 */
public class TransitionalPageFactory extends AbstractUserDirectoryPageFactory implements AmetysObjectFactory<TransitionalPage>
{
    /**
     * Create a transitional page.
     * @param root the user directory root page.
     * @param prefix the page's title.
     * @param path the path
     * @return The TransitionalPage created
     */
    public TransitionalPage createTransitionalPage(Page root, String prefix, String path)
    {
        return new TransitionalPage(root, getConfiguration(), getScheme(), this, prefix, path);
    }
    
    public TransitionalPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // E.g: udtransitional://path?rootId=...
        String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "udtransitional://"), "?rootId=");
        
        String rootId = StringUtils.substringAfter(id, "?rootId=");
        Page root = _resolver.resolveById(rootId);
        
        String pathName = path;
        String parentPath = "_root";
        if (StringUtils.contains(path, "/"))
        {
            pathName = StringUtils.substringAfterLast(path, "/");
            parentPath = StringUtils.substringBeforeLast(path, "/");
        }
        
        String name = _userDirectoryPageHandler.getName(pathName);
        
        SortedSet<String> transitionalPagesName = _userDirectoryPageHandler.getTransitionalPagesName(root, _userDirectoryPageHandler.getName(parentPath));
        if (transitionalPagesName.contains(name))
        {
            return createTransitionalPage(root, name, path);
        }
        else
        {
            throw new UnknownAmetysObjectException("No transitional page named " + name + " (full page path " + path + ").");
        }
    }

    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        try
        {
            // E.g: udtransitional://path?rootId=...
            String path = StringUtils.substringBefore(StringUtils.substringAfter(id, "udtransitional://"), "?rootId=");
            
            String rootId = StringUtils.substringAfter(id, "?rootId=");
            Page root = _resolver.resolveById(rootId);
            
            String pathName = path;
            String parentPath = "_root";
            if (StringUtils.contains(path, "/"))
            {
                pathName = StringUtils.substringAfterLast(path, "/");
                parentPath = StringUtils.substringBeforeLast(path, "/");
            }
            
            String name = _userDirectoryPageHandler.getName(pathName);
            
            SortedSet<String> transitionalPagesName = _userDirectoryPageHandler.getTransitionalPagesName(root, _userDirectoryPageHandler.getName(parentPath));
            return transitionalPagesName.contains(name);
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
        
    }

    public String getScheme()
    {
        return "udtransitional";
    }
}
