/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;

/**
 * Resolves an user directory page path from the associated user content.
 */
public class UserDirectoryPageResolver extends AbstractLogEnabled implements Component, Serviceable, Initializable, Contextualizable
{
    /** The avalon role. */
    public static final String ROLE = UserDirectoryPageResolver.class.getName();
    
    /** The key cache for empty site */
    protected static final String _CACHE_KEY_EMPTY_SITE = "_ametys_no_site";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    /** The user directory page handler */
    protected UserDirectoryPageHandler _pageHandler;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The context */
    protected Context _context;
    /** The languages manager */
    protected LanguagesManager _languagesManager;
    
    static class UserPageElementKey extends AbstractCacheKey
    {
        UserPageElementKey(String lang, String contentId, String siteName, String contentTypeId)
        {
            super(lang, contentId, siteName, contentTypeId);
        }
        
        static UserPageElementKey of(String lang, String contentId, String siteName, String contentTypeId)
        {
            return new UserPageElementKey(lang, contentId, siteName, contentTypeId);
        }
        
        static UserPageElementKey of(String lang)
        {
            return new UserPageElementKey(lang, null, null, null);
        } 
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _pageHandler = (UserDirectoryPageHandler) serviceManager.lookup(UserDirectoryPageHandler.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
        _languagesManager = (LanguagesManager) serviceManager.lookup(LanguagesManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(ROLE, 
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_USER_PAGE_CACHE_LABEL"),
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_USER_PAGE_CACHE_DESCRIPTION"),
                true,
                null);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Return the user page
     * @param userContent the user page content
     * @param siteName The current site name.
     * @param sitemapName The current sitemap name.
     * @return the user page or <code>null</code> if the user page does not exist
     */
    public UserPage getUserPage(Content userContent, String siteName, String sitemapName)
    {
        for (String contentTypeId : userContent.getTypes())
        {
            UserPage userPage = getUserPage(userContent, siteName, sitemapName, contentTypeId);
            if (userPage != null)
            {
                return userPage;
            }
        }
        
        return null;
    }
    
    /**
     * Return the user page
     * @param userContent the user page content
     * @param siteName The current site name.
     * @param sitemapName The current sitemap name.
     * @param contentTypeId the content type id
     * @return the user page or <code>null</code> if the user page does not exist
     */
    public UserPage getUserPage(Content userContent, String siteName, String sitemapName, String contentTypeId)
    {
        String siteKey = StringUtils.isNotBlank(siteName) ? siteName : _CACHE_KEY_EMPTY_SITE;
        String pageId = null;
        String contentLang = userContent.getLanguage();
        if (StringUtils.isNotBlank(contentLang))
        {
            // Just get the user page of content language
            UserPageElementKey key = UserPageElementKey.of(contentLang, userContent.getId(), siteKey, contentTypeId);
            pageId = _getCache().get(key, k -> _getUserPageId(userContent, siteName, contentLang, contentTypeId));
        }
        else
        {
            // The content is multilingual, so test different languages starting with the sitemap name to get an user content page
            Iterator<String> languageIterator = _getLanguagesToTest(sitemapName).iterator();
            while (pageId == null && languageIterator.hasNext())
            {
                String lang = languageIterator.next();
                UserPageElementKey key = UserPageElementKey.of(lang, userContent.getId(), siteKey, contentTypeId);
                pageId = _getCache().get(key, k -> _getUserPageId(userContent, siteName, lang, contentTypeId));
            }
        }
        
        return _getUserPage(pageId);
    }
    
    private Set<String> _getLanguagesToTest(String givenLang)
    {
        Set<String> languages = new HashSet<>();
        
        // First, test the given language if not blank
        if (StringUtils.isNotBlank(givenLang))
        {
            languages.add(givenLang);
        }
        
        // Then test english language
        languages.add("en");
        
        // Finaly test other languages
        languages.addAll(_languagesManager.getAvailableLanguages().keySet());
        
        return languages;
    }
    
    /**
     * Return the user page id
     * @param userContent the user content
     * @param siteName the site name
     * @param language the given language
     * @param contentTypeId the content type id
     * @return the user page id or <code>null</code> if the user page does not exist
     */
    protected String _getUserPageId(Content userContent, String siteName, String language, String contentTypeId)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Use default workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Page userDirectoryRootPage = _pageHandler.getUserDirectoryRootPage(siteName, language, contentTypeId);
            if (userDirectoryRootPage == null)
            {
                return null;
            }
            
            return getUserPageId(userDirectoryRootPage, userContent);
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Return the user page
     * @param userDirectoryRootPage the user directory root page
     * @param userContent the user content
     * @return the user page or <code>null</code> if the user page does not exist
     */
    public UserPage getUserPage(Page userDirectoryRootPage, Content userContent)
    {
        String pageId = getUserPageId(userDirectoryRootPage, userContent);
        return _getUserPage(pageId);
    }
    
    /**
     * Return the user page identifier
     * @param userDirectoryRootPage the user directory root page
     * @param userContent the user content
     * @return the user page or <code>null</code> if the user page does not exist
     */
    public String getUserPageId(Page userDirectoryRootPage, Content userContent)
    {
        // E.g: uduser://path?rootId=...&contentId=...
        String userPath = _getUserPath(userDirectoryRootPage, userContent);
        if (userPath != null)
        {
            return UserPage.getId(userPath, userDirectoryRootPage.getId(), userContent.getId());
        }
        return null;
    }
    
    /**
     * Get the path of a user content
     * @param userDirectoryRootPage The user directory root page where to look into
     * @param userContent The user content
     * @return The path or <code>null</code> if the user page does not exist
     */
    private String _getUserPath(Page userDirectoryRootPage, Content userContent)
    {
        String value = _pageHandler.getTransformedClassificationValue(userDirectoryRootPage, userContent);
        return value != null ? value.replaceAll("(.)(?!$)", "$1/") : null;
    }
    
    private UserPage _getUserPage(String pageId)
    {
        try
        {
            return pageId != null ? _ametysResolver.resolveById(pageId) : null;
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    private Cache<UserPageElementKey, String> _getCache()
    {
        return _cacheManager.get(ROLE);
    }
    
    /**
     * Invalidate the user page cache for one language. If lang is <code>null</code>, invalidate all
     * @param lang the lang. Can be <code>null</code>
     */
    public void invalidateUserPageCache(String lang)
    {
        if (StringUtils.isNotBlank(lang)) 
        {
            _getCache().invalidate(UserPageElementKey.of(lang));
        }
        else 
        {
            _getCache().invalidateAll();
        }
    }
}
