/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.repository.ACLAmetysObject;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.AbstractConfigurableVirtualPage;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Page representing a second-level page.
 */
public class UserPage extends AbstractConfigurableVirtualPage<UserPageFactory> implements ACLAmetysObject
{
    private int _initialDepth;
    private String _title;
    private String _path;
    private Content _syncContent;
    
    /**
     * Constructor.
     * @param root the root page.
     * @param syncContent the synchronized content
     * @param path the path
     * @param configuration The abstract virtual page's configuration
     * @param scheme The scheme
     * @param factory The user page factory
     */
    public UserPage(Page root, VirtualPageConfiguration configuration, String scheme, UserPageFactory factory, Content syncContent, String path)
    {
        super(root, configuration, scheme, factory);
        
        _path = path;
        _syncContent = syncContent;
        
        _title = _syncContent.getTitle(new Locale(root.getSitemapName()));
        _initialDepth = _factory.getUserDirectoryPageHandler().getDepth(_root);
    }

    /**
     * Compute a page id
     * @param path The path
     * @param rootId The root page id
     * @param contentId The content id
     * @return The id
     */
    public static String getId(String path, String rootId, String contentId)
    {
        // E.g: uduser://path?rootId=...&contentId=...
        return "uduser://" + (StringUtils.isEmpty(path) ? "_root" : path) + "?rootId=" + rootId + "&contentId=" + contentId;
    }
    
    /**
     * Returns the associated synchronizable {@link Content}.
     * @return the associated synchronizable {@link Content}.
     */
    @SuppressWarnings("unchecked")
    @Override
    public Content getContent()
    {
        return _syncContent;
    }
    
    
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + _initialDepth + 1;
    }

    @Override
    public Set<String> getReferers() throws AmetysRepositoryException
    {
        return null;
    }

    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        return new CollectionIterable<>(children);
    }

    
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        if (_path.equals("_root"))
        {
            return _root.getPathInSitemap() + "/" + getName();
        }
        else
        {
            String path = StringUtils.lowerCase(_path);
            return _root.getPathInSitemap() + "/" + path + "/" + getName();
        }
    }
    
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        return null;
    }

    @SuppressWarnings("unchecked")
    
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return false;
    }
    
    
    public String getId() throws AmetysRepositoryException
    {
        return getId(_path, _root.getId(), _syncContent.getId());
    }

    
    public String getName() throws AmetysRepositoryException
    {
        return _syncContent.getName();
    }
    
    @SuppressWarnings("unchecked")
    
    public Page getParent() throws AmetysRepositoryException
    {
        if (_initialDepth > 0)
        {
            String pathName = StringUtils.substringAfterLast(_path, "/");
            String name = _factory.getUserDirectoryPageHandler().getName(pathName);
            
            return _factory.getTransitionalPageFactory().createTransitionalPage(_root, name, _path);
        }
        else
        {
            return _root;
        }
    }

    
    public String getParentPath() throws AmetysRepositoryException
    {
        if (_initialDepth > 0)
        {
            String path = StringUtils.lowerCase(_path);
            return _root.getPath() + "/" + path;
        }
        else
        {
            return _root.getPath();
        }
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }
    
    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePage) throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        return new CollectionIterable<>(children);
    }

    @Override
    public boolean isVisible() throws AmetysRepositoryException
    {
        return false;
    }

    
    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        throw new UnknownAmetysObjectException("There is no child for user page");
    }

    public Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser()
    {
        Set<String> collectionIds = _factory.getSccHelper().getSynchronizableCollectionIds(_syncContent);
        for (String collectionId : collectionIds)
        {
            SynchronizableContentsCollection collection = _factory.getSynchronizableContentsCollectionDAO().getSynchronizableContentsCollection(collectionId);
            if (collection != null)
            {
                String restrictedField = collection.getRestrictedField();
                if (!StringUtils.isEmpty(restrictedField))
                {
                    Boolean restricted = _syncContent.getValue(restrictedField);
                    if (restricted != null && restricted.booleanValue())
                    {
                        return Map.of(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED, Set.of(RightManager.READER_PROFILE_ID));
                    }
                }
            }
        }
        
        return Map.of();
    }
    
    public Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Set<GroupIdentity> groups)
    {
        return Map.of();
    }
    
    public Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(UserIdentity user)
    {
        return Map.of();
    }
    
    public boolean isInheritanceDisallowed()
    {
        return false;
    }
}
