/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.action;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ExplorerFile;
import org.ametys.cms.data.File;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.transformation.xslt.ProfileImageResolverHelper;

/**
 * Get the internal uri of a user's avatar
 *
 */
public class GetProfileImageUriAction extends ServiceableAction
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
    }
    
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        UserIdentity user = _getUser(parameters);
        
        int size = parameters.getParameterAsInteger("size", 0);
        boolean download = parameters.getParameterAsBoolean("download", false);
        String lang = parameters.getParameter("lang");
        
        result.put("imageUri", _getProfileImageUri(user, lang, size, download));
        
        return result;
    }
    
    /**
     * Get the profile image uri
     * @param user the user identity. Can be null.
     * @param lang the language
     * @param cropSize the image dimension. The image will be cropped.
     * @param download true if the uri is for download purposes.
     * @return the profile image uri
     * @throws ResourceNotFoundException if no image for user was found
     */
    protected String _getProfileImageUri(UserIdentity user, String lang, int cropSize, boolean download) throws ResourceNotFoundException
    {
        if (user == null)
        {
            throw new ResourceNotFoundException("Unable to get a profile image for a null user");
        }
        else
        {
            String imageUri = _getProfileImageUriFromUserContentIfExists(user, lang, cropSize, download);
            if (imageUri == null)
            {
                imageUri = _getDefaultImageUri(user, lang, cropSize, download);
            }
            
            return imageUri;
        }
    }
    
    private String _getProfileImageUriFromUserContentIfExists(UserIdentity user, String lang, int cropSize, boolean download)
    {
        Content userContent = _userDirectoryHelper.getUserContent(user, lang);
        if (userContent != null && userContent.hasValue(ProfileImageResolverHelper.USER_CONTENT_IMAGE_PATH))
        {
            File imgFile = userContent.getValue(ProfileImageResolverHelper.USER_CONTENT_IMAGE_PATH);
            if (imgFile instanceof ExplorerFile explorerFile)
            {
                return ResolveURIComponent.resolveCroppedImage("explorer", explorerFile.getResourceId(), cropSize, cropSize, download, false, true);
            }
            else
            {
                // Binary
                String imgUri = ProfileImageResolverHelper.USER_CONTENT_IMAGE_PATH + "?contentId=" + userContent.getId();
                return ResolveURIComponent.resolveCroppedImage("attribute", imgUri, cropSize, cropSize, download, false, true);
            }
        }
        
        return null;
    }
    
    /**
     * Get the default image uri for a user
     * @param user the user identity
     * @param lang the content language
     * @param cropSize the image dimension
     * @param download true if the uri is for download purposes.
     * @return the image uri
     */
    protected String _getDefaultImageUri(UserIdentity user, String lang, int cropSize, boolean download)
    {
        StringBuilder sb = new StringBuilder("cocoon://_plugins/core-ui/user/");
        sb.append(user.getPopulationId())
            .append("/")
            .append(user.getLogin())
            .append("/image_")
            .append(cropSize);
        
        List<String> params = new ArrayList<>();
        if (download)
        {
            params.add("download=true");
        }
        
        if (params.size() > 0)
        {
            sb.append("?").append(String.join("&", params));
        }
        
        return sb.toString();
    }
    
    /**
     * Get the user from parameters
     * @param parameters the sitemap parameters
     * @return the user identity
     */
    protected UserIdentity _getUser(Parameters parameters)
    {
        String login =  parameters.getParameter("login", StringUtils.EMPTY);
        String populationId =  parameters.getParameter("populationId", StringUtils.EMPTY);
        // Default to current user if login not provided, except for the default source for which a login is not needed.
        if (StringUtils.isEmpty(login) || StringUtils.isEmpty(populationId))
        {
            return _currentUserProvider.getUser();
        }
        else
        {
            return new UserIdentity(login, populationId);
        }
    }
}
