/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.generator;

import java.io.IOException;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generate information to render the organization chart service.
 */
public class OrgChartGenerator extends ServiceableGenerator
{
    private static final String __VIEW_NAME = "organizationChart";
    
    private OrganisationChartPageHandler _organizationChartPageHandler;
    private SourceResolver _sourceResolver;
    private ContentHelper _contentHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _organizationChartPageHandler = (OrganisationChartPageHandler) serviceManager.lookup(OrganisationChartPageHandler.ROLE);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String language = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "orgUnits");
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        String contentTypeId = zoneItem.getServiceParameters().getValue("content-type");
        List<Content> contents = _organizationChartPageHandler.getFirstLevelOfContents(language, contentTypeId).toList();
        for (Content content : contents)
        {
            saxOrganizationUnit(content);
        }

        XMLUtils.endElement(contentHandler, "orgUnits");
        contentHandler.endDocument();
    }
    
    /**
     * SAX an organization unit content
     * @param orgUnit the organization unit to sax.
     * @throws SAXException if an error occurs
     */
    protected void saxOrganizationUnit(Content orgUnit) throws SAXException
    {
        SitemapSource src = null;
        try
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", orgUnit.getId());
            attrs.addCDATAAttribute("unique-id", org.ametys.core.util.StringUtils.md5Base64(orgUnit.getId()));
            attrs.addCDATAAttribute("name", orgUnit.getName());
            
            XMLUtils.startElement(contentHandler, "orgUnit", attrs);
            
            String format = parameters.getParameter("output-format", "html");
            String uri = _contentHelper.getContentViewUrl(orgUnit, __VIEW_NAME, format);
            src = (SitemapSource) _sourceResolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(contentHandler));
            
            saxFirstUserOfOrganizationUnit(orgUnit);
            saxOrganizationUnitChildren(orgUnit);
            
            XMLUtils.endElement(contentHandler, "orgUnit");
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }
    
    /**
     * SAX the first user with a role in the organization unit content
     * @param orgUnit the organization unit
     * @throws SAXException if an error occurred
     */
    protected void saxFirstUserOfOrganizationUnit(Content orgUnit) throws SAXException
    {
        if (orgUnit.hasValue("users"))
        {
            ModelAwareRepeater repeaters = orgUnit.getRepeater("users");
            for (ModelAwareRepeaterEntry entry : repeaters.getEntries())
            {
                String role = entry.getValue("role");
                ContentValue userValue = entry.getValue("user");
                if (StringUtils.isNotEmpty(role) && userValue != null)
                {
                    try
                    {
                        Content user = userValue.getContent();
                        saxUser(user, role);
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        getLogger().error("The entity " + orgUnit.getTitle() + " (" + orgUnit.getId() + ") is referencing an unexisting user '" + userValue.getContentId() + "'");
                    }
                    catch (Exception e)
                    {
                        getLogger().error("Fail to sax main user '" + userValue.getContentId() + "' for entity " + orgUnit.getTitle() + " (" + orgUnit.getId() + ")", e);
                    }
                    
                    break;
                }
            }
        }
    }
    
    /**
     * SAX a user content
     * @param user the user to sax.
     * @param role the user's role
     * @throws SAXException if an error occurs
     */
    protected void saxUser(Content user, String role) throws SAXException
    {
        SitemapSource src = null;
        try
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", user.getId());
            attrs.addCDATAAttribute("name", user.getName());
            
            XMLUtils.startElement(contentHandler, "main-user", attrs);
            
            XMLUtils.createElement(contentHandler, "role", role);
            
            String format = parameters.getParameter("output-format", "html");
            String uri = _contentHelper.getContentViewUrl(user, __VIEW_NAME , format);
            src = (SitemapSource) _sourceResolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(contentHandler));
            
            XMLUtils.endElement(contentHandler, "main-user");
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }
    
    /**
     * SAX children of an organization unit
     * @param orgUnit the organization unit
     * @throws SAXException if an error occurred
     */
    protected void saxOrganizationUnitChildren(Content orgUnit) throws SAXException
    {
        List<Content> children = _organizationChartPageHandler.getChildContents(orgUnit);

        if (!children.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "orgUnits");

            for (Content child : children)
            {
                saxOrganizationUnit(child);
            }

            XMLUtils.endElement(contentHandler, "orgUnits");
        }
    }
}
