/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.userdirectory.observation;

import java.util.Map;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Context;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.page.OrgUnitPage;
import org.ametys.plugins.userdirectory.page.OrganisationChartPageResolver;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.plugins.userdirectory.page.VirtualOrganisationChartPageFactory;
import org.ametys.plugins.userdirectory.page.VirtualUserDirectoryPageFactory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * Abstract {@link Observer} for observing validation of User content.
 */
public abstract class AbstractContentObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The context. */
    protected org.apache.avalon.framework.context.Context _context;
    /** Cocoon context. */
    protected Context _cocoonContext;
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The content type helper */
    protected ContentTypesHelper _contentTypeHelper;
    /** The page element cache */
    protected PageElementCache _zoneItemCache;
    /** The resolver for user directory pages */
    protected UserDirectoryPageResolver _userDirectoryPageResolver;
    /** The resolver for ud orgunits pages */
    protected OrganisationChartPageResolver _organisationChartPageResolver;

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(org.apache.cocoon.Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _zoneItemCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/zoneItem");
        _userDirectoryPageResolver = (UserDirectoryPageResolver) manager.lookup(UserDirectoryPageResolver.ROLE);
        _organisationChartPageResolver = (OrganisationChartPageResolver) manager.lookup(OrganisationChartPageResolver.ROLE);
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars)
    {
        try
        {
            Content content = _getTarget(event);
            if (_isUserContent(content))
            {
                AmetysObjectIterable<Page> rootPages = _getUserRootPages ();
                if (!rootPages.iterator().hasNext())
                {
                    getLogger().debug("There's no user root page, nothing to invalidate");
                    return;
                }
                
                for (Page rootPage : rootPages)
                {
                    String cType = rootPage.getValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME);
                    if (ArrayUtils.contains(content.getTypes(), cType))
                    {
                        _internalObserve(event, rootPage, content);
                    }
                }
            }
            else if (_isOrgUnitContent(content))
            {
                AmetysObjectIterable<Page> rootPages = _getOrgUnitRootPages ();
                if (!rootPages.iterator().hasNext())
                {
                    getLogger().debug("There's no orgUnit root page, nothing to invalidate");
                    return;
                }
                
                for (Page rootPage : rootPages)
                {
                    _internalObserve(event, rootPage, content);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Unable to observe event: " + event, e);
        }
    }
    
    /**
     * Do the actual work.
     * @param event the observation event.
     * @param rootUsersPage the page holding the virtual user pages
     * @param userContent a list containing all impacted user contents.
     */
    protected abstract void _internalObserve(Event event, Page rootUsersPage, Content userContent);
    
    /**
     * Get the user root pages
     * @return the user root pages
     */
    protected AmetysObjectIterable<Page> _getUserRootPages ()
    {
        Expression expression = new VirtualFactoryExpression(VirtualUserDirectoryPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
    
    /**
     * Get the orgUnit root pages
     * @return the orgUnit root pages
     */
    protected AmetysObjectIterable<Page> _getOrgUnitRootPages ()
    {
        Expression expression = new VirtualFactoryExpression(VirtualOrganisationChartPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
    
    /**
     * Retrieve the target of the observer
     * @param event The event
     * @return The target
     * @throws Exception if failed to get content
     */
    protected Content _getTarget(Event event) throws Exception
    {
        return (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
    }
    
    /**
     * Return true if the content is a user content
     * @param content the content
     * @return true if the content is a user content
     */
    protected boolean _isUserContent(Content content)
    {
        return content != null && _contentTypeHelper.isInstanceOf(content, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
    }
    
    /**
     * Return true if the content is a orgUnit content
     * @param content the content
     * @return true if the content is a orgUnit content
     */
    protected boolean _isOrgUnitContent(Content content)
    {
        return content != null && _contentTypeHelper.isInstanceOf(content, UserDirectoryHelper.ORGUNIT_CONTENT_TYPE);
    }
    
    /**
     * Remove zone item cache
     * @param rootPage the root page
     * @param content the content 
     * @param workspace the workspace
     */
    protected void _removeZoneItemCache(Page rootPage, Content content, String workspace)
    {
        if (_isUserContent(content))
        {
            UserPage userPage = _userDirectoryPageResolver.getUserPage(rootPage, content);
            if (userPage != null)
            {
                AmetysObjectIterable< ? extends Zone> zones = userPage.getZones();
                for (Zone zone : zones)
                {
                    for (ZoneItem zoneItem : zone.getZoneItems())
                    {
                        if (zoneItem.getType().equals(ZoneType.CONTENT)) 
                        {
                            _zoneItemCache.removeItem(workspace, rootPage.getSiteName(), "CONTENT", zoneItem.getId());
                        }
                    }
                }
            }
        }
        else if (_isOrgUnitContent(content))
        {
            OrgUnitPage orgUnitPage = _organisationChartPageResolver.getOrgUnitPage(rootPage, content);
            if (orgUnitPage != null)
            {
                AmetysObjectIterable< ? extends Zone> zones = orgUnitPage.getZones();
                for (Zone zone : zones)
                {
                    for (ZoneItem zoneItem : zone.getZoneItems())
                    {
                        if (zoneItem.getType().equals(ZoneType.CONTENT)) 
                        {
                            _zoneItemCache.removeItem(workspace, rootPage.getSiteName(), "CONTENT", zoneItem.getId());
                        }
                    }
                }
            }
        }
    }
}
