/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.AbstractConfigurableVirtualPage;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;


/**
 * Page representing a second-level page.
 */
public class TransitionalPage extends AbstractConfigurableVirtualPage<TransitionalPageFactory>
{
    private static Logger _logger = LoggerFactory.getLogger(TransitionalPage.class);
    private String _path;
    private int _initialDepth;
    private String _prefix;
    
    /**
     * Constructor.
     * @param root the user directory root page.
     * @param prefix the page's title.
     * @param path the path
     * @param configuration The abstract virtual page configuration
     * @param scheme The scheme
     * @param factory The transitional page factory
     */
    public TransitionalPage(Page root, VirtualPageConfiguration configuration, String scheme, TransitionalPageFactory factory, String prefix, String path)
    {
        super(root, configuration, scheme, factory);
        
        _prefix = prefix;
        _path = path;
        
        _initialDepth = _factory.getUserDirectoryPageHandler().getDepth(_root);
    }
    
    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + _path.split("/").length;
    }

    @Override
    public Set<String> getReferers() throws AmetysRepositoryException
    {
        return null;
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return StringUtils.upperCase(_prefix);
    }
    
    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return StringUtils.upperCase(_prefix);
    }

    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();

        int depth = _initialDepth - _path.split("/").length;
        if (depth > 0)
        {
            SortedSet<String> transitionalPagesName = _factory.getUserDirectoryPageHandler().getTransitionalPagesName(_root, _factory.getUserDirectoryPageHandler().getName(_path));
            for (String name : transitionalPagesName)
            {
                String pathName = _factory.getUserDirectoryPageHandler().getPathName(name);
                children.add(_factory.createTransitionalPage(_root, name, _path + "/" + pathName));
            }
            
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, _factory.getUserDirectoryPageHandler().getName(_path));
            for (Entry<String, String> entry : userPagesContent.entrySet())
            {
                String contentTypeId = _factory.getUserDirectoryPageHandler().getContentTypeId(_root);
                String classificationAttribute = _factory.getUserDirectoryPageHandler().getClassificationAttribute(_root);
                Content content;
                try
                {
                    content = _factory.getResolver().resolveById(entry.getValue());
                }
                catch (AmetysRepositoryException e)
                {
                    // content does not exist, skip to next iteration
                    break;
                }
                if (content == null || !Arrays.asList(content.getTypes()).contains(contentTypeId) || !content.hasValue(classificationAttribute))
                {
                    break;
                }
                
                String classificationAttributeValue = _factory.getUserDirectoryPageHandler().getTransformedClassificationValue(_root, content);
                if (classificationAttributeValue != null && classificationAttributeValue.length() == _path.split("/").length)
                {
                    children.add(_factory.getUserPageFactory().createUserPage(_root, content, _path));
                }
            }
        }
        else
        {
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, _path);
            for (String contentId : userPagesContent.values())
            {
                try
                {
                    Content content = _factory.getResolver().resolveById(contentId);
                    children.add(_factory.getUserPageFactory().createUserPage(_root, content, _path));
                }
                catch (UnknownAmetysObjectException e)
                {
                    _logger.warn("Content {} does not exist anymore", contentId, e);
                }
            }
        }
        
        return new CollectionIterable<>(children);
    }

    @Override
    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePage) throws AmetysRepositoryException
    {
        if (includeInvisiblePage)
        {
            return getChildrenPages();
        }
        else
        {
            List<Page> children = new ArrayList<>();
            return new CollectionIterable<>(children);
        }
    }
    
    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        String path = StringUtils.lowerCase(_path);
        return _root.getPathInSitemap() + "/" + path;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        String completePath = _path + "/" + path;
        int depth = _initialDepth - completePath.split("/").length + 1;
        if (depth > 0)
        {
            String namePath = StringUtils.substringAfterLast(completePath, "/");
            String parentPath = StringUtils.substringBeforeLast(completePath, "/");

            SortedSet<String> transitionalPagesName = _factory.getUserDirectoryPageHandler().getTransitionalPagesName(_root, parentPath);
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, parentPath);
            String name = _factory.getUserDirectoryPageHandler().getName(namePath);
            if (transitionalPagesName.contains(name))
            {
                TransitionalPage page = _factory.createTransitionalPage(_root, name, completePath);
                return (A) page;
            }
            else if (userPagesContent.containsKey(name))
            {
                String contentId = userPagesContent.get(name);
                Content syncContent = _factory.getResolver().resolveById(contentId);
                UserPage page = _factory.getUserPageFactory().createUserPage(_root, syncContent, parentPath);
                return (A) page;
            }
            else
            {
                throw new UnknownAmetysObjectException("No transitional page named " + name + " (full page path " + path + ").");
            }
        }
        else
        {
            String userPath = StringUtils.substringBeforeLast(completePath, "/");
            String contentName = StringUtils.substringAfterLast(completePath, "/");
            
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, userPath);
            if (userPagesContent.containsKey(contentName))
            {
                Content content = _factory.getResolver().resolveById(userPagesContent.get(contentName));
                
                UserPage page = _factory.getUserPageFactory().createUserPage(_root, content, userPath);
                return (A) page;
            }
            else
            {
                throw new UnknownAmetysObjectException("No user content named " + contentName + " (full page path " + path + ").");
            }
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        int depth = _initialDepth - _path.split("/").length;
        if (depth > 0)
        {
            SortedSet<String> transitionalPagesName = _factory.getUserDirectoryPageHandler().getTransitionalPagesName(_root, _path);
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, _path);
            return transitionalPagesName.contains(name) || userPagesContent.containsKey(name);
        }
        else
        {
            Map<String, String> userPagesContent = _factory.getUserDirectoryPageHandler().getUserPagesContent(_root, _path);
            return userPagesContent.containsKey(name);
        }
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: udtransitional://path?rootId=...
        return "udtransitional://" + _path + "?rootId=" + _root.getId();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        return StringUtils.lowerCase(_prefix);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        if (_path.split("/").length > 1)
        {
            String parentPath = StringUtils.substringBeforeLast(_path, "/");
            String pathName = parentPath;
            if (StringUtils.contains(pathName, "/"))
            {
                pathName = StringUtils.substringAfterLast(pathName, "/");
            }
            
            String name = _factory.getUserDirectoryPageHandler().getName(pathName);
            return _factory.createTransitionalPage(_root, name, parentPath);
        }
        else
        {
            return _root;
        }
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        if (_path.split("/").length > 1)
        {
            String path = StringUtils.lowerCase(_path);
            return _root.getPath() + "/" + StringUtils.substringBeforeLast(path, "/");
        }
        else
        {
            return _root.getPath();
        }
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    @Override
    public boolean isVisible() throws AmetysRepositoryException
    {
        return false;
    }

    @Override
    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        return getChildrenPages().stream().collect(Collectors.toList()).get(index);
    }
}
