/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} for handling "virtual" organisation chart root page
 */
public class VirtualOrganisationChartPageFactory extends AbstractUserDirectoryPageFactory implements VirtualAmetysObjectFactory<Page>
{
    private OrganisationChartPageHandler _organisationChartPageHandler;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);

        _organisationChartPageHandler = (OrganisationChartPageHandler) manager.lookup(OrganisationChartPageHandler.ROLE);
    }

    @Override
    public Page getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public String getScheme()
    {
        return "ocroot";
    }

    @Override
    public AmetysObjectIterable<Page> getChildren(JCRAmetysObject parent)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the organisation chart virtual pages should be a page.");
        }
        
        List<Page> children = new ArrayList<>();
        Page rootPage = (Page) parent;
        
        _organisationChartPageHandler.getContentsForRootPage(rootPage)
            .forEach(content -> children.add(getOrgUnitPageFactory().createOrgUnitPage(rootPage, content, content.getName())));
        
        return new CollectionIterable<>(children);
    }

    @Override
    public Page getChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the organisation chart virtual pages should be a page.");
        }
        
        Page rootPage = (Page) parent;
        Stream<Content> contents = _organisationChartPageHandler.getContentsForRootPage(rootPage);
        List<Content> contentFilters = contents.filter(c -> c.getName().equals(childName)).collect(Collectors.toList());
        
        if (!contentFilters.isEmpty())
        {
            Content content = contentFilters.get(0);
            return getOrgUnitPageFactory().createOrgUnitPage(rootPage, content, content.getName());
        }
        else
        {
            throw new UnknownAmetysObjectException("No orgUnit page named " + childName);
        }
    }

    @Override
    public boolean hasChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the organisation chart virtual pages should be a page.");
        }
        
        Page rootPage = (Page) parent;
        Stream<Content> contents = _organisationChartPageHandler.getContentsForRootPage(rootPage);
        List<Content> contentFilters = contents.filter(c -> c.getName().equals(childName)).collect(Collectors.toList());
        
        return !contentFilters.isEmpty();
    }
}
