/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.observation;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.web.repository.page.Page;

/**
 * {@link Observer} when a user content is modified, deleted or validated.
 * Clear the cache of virtual user pages and reindex all virtual pages on modification.
 */
public class UserContentCacheObserver extends AbstractContentObserver
{
    /** The user directory page handler. */
    protected UserDirectoryPageHandler _userPageHandler;
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userPageHandler = (UserDirectoryPageHandler) manager.lookup(UserDirectoryPageHandler.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
    }

    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING)  
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE) 
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED) 
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_ADDED) 
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED);
    }

    @Override
    public int getPriority()
    {
        // Will be processed AFTER live synchronization observers and AFTER page unindexing observer
        return MAX_PRIORITY + 5000;
    }

    @Override
    protected void _internalObserve(Event event, Page rootUsersPage, Content userContent)
    {
        if (_isUserContent(userContent))
        {
            if (!event.getId().equals(ObservationConstants.EVENT_CONTENT_ADDED))
            {
                _userPageHandler.clearCache(rootUsersPage);
                // Invalidate user page cache
                _userDirectoryPageResolver.invalidateUserPageCache(userContent.getLanguage());
            }
            
            if (!event.getId().equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE) && !event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED))
            {
                // Invalidate user content cache
                _userDirectoryHelper.invalidateUserContentCache(userContent.getLanguage());
            }
        }
    }
}
