/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.service.search;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.query.Query;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.web.frontoffice.search.instance.model.SearchContext;
import org.ametys.web.frontoffice.search.instance.model.SiteContext;
import org.ametys.web.frontoffice.search.instance.model.SitemapContext;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.metamodel.Returnable;
import org.ametys.web.frontoffice.search.metamodel.ReturnableExtensionPoint;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.PageReturnable;
import org.ametys.web.frontoffice.search.metamodel.impl.PageSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.PrivateContentReturnable;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.search.query.ContentPageQuery;

/**
 * The returnable for users' pages
 */
public class UserPageReturnable extends PrivateContentReturnable
{
    /** The returnable for pages. Only used because {@link UserSaxer} extends {@link PageSaxer} whose constructor needs it. */
    protected PageReturnable _pageReturnable;
    /** The helper for content types */
    protected ContentTypesHelper _contentTypeHelper;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    /** The resolver for user directory virtual pages */
    protected UserDirectoryPageResolver _userDirectoryPageResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        ReturnableExtensionPoint returnableEP = (ReturnableExtensionPoint) manager.lookup(ReturnableExtensionPoint.ROLE);
        _pageReturnable = (PageReturnable) returnableEP.getExtension(PageReturnable.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _userDirectoryPageResolver = (UserDirectoryPageResolver) manager.lookup(UserDirectoryPageResolver.ROLE);
    }
    
    @Override
    protected Function<Query, Query> siteQueryJoiner()
    {
        // User contents are created through plugin content-io, and are "off-site". 
        // When selecting a site context, it should search that a linked page exist in the selected site
        return ContentPageQuery::new;
    }
    
    @Override
    public ReturnableSaxer getSaxer(Collection<Returnable> allReturnables, AdditionalParameterValueMap additionalParameterValues)
    {
        return new UserSaxer(additionalParameterValues);
    }
    
    /**
     * The {@link ReturnableSaxer} for user contents.
     * <br>Their pages are SAXed.
     */
    public class UserSaxer extends PageSaxer
    {
        /** The user content types selected in the additional parameter of the current servicee instance (and their descendants) */
        protected Collection<String> _allContentTypes;
        
        /**
         * Constructor
         * @param additionalParameterValues The additional parameter values
         */
        public UserSaxer(AdditionalParameterValueMap additionalParameterValues)
        {
            super(UserPageReturnable.this._pageReturnable);
            _allContentTypes = _allContentTypeIds(additionalParameterValues);
        }
        
        @SuppressWarnings("synthetic-access")
        private Collection<String> _currentServiceInstanceContentTypes(AdditionalParameterValueMap additionalParameterValues)
        {
            return UserPageReturnable.this.getContentTypeIds(additionalParameterValues);
        }
        
        private Collection<String> _allContentTypeIds(AdditionalParameterValueMap additionalParameterValues)
        {
            return _currentServiceInstanceContentTypes(additionalParameterValues)
                    .stream()
                    .map(this::_selfAndDescendantContentTypes)
                    .flatMap(Set::stream)
                    .collect(Collectors.toList());
        }
        
        private Set<String> _selfAndDescendantContentTypes(String cType)
        {
            return SetUtils.union(
                    Collections.singleton(cType), 
                    _contentTypeExtensionPoint.getSubTypes(cType));
        }
        
        @Override
        public String getIdentifier()
        {
            return PageSaxer.class.getName();
        }
        
        @Override
        public boolean canSax(AmetysObject hit, Logger logger, SearchComponentArguments args)
        {
            return hit instanceof Content
                    && _contentTypeHelper.isInstanceOf((Content) hit, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
        }
        
        @Override
        public void sax(ContentHandler contentHandler, AmetysObject hit, Logger logger, SearchComponentArguments args) throws SAXException
        {
            Content userContent = (Content) hit;
            SearchContext searchContext = _getSearchContext(args);
            String siteName = _getSiteName(args, searchContext);
            String sitemapName = _getSitemapName(args, searchContext, userContent);
            UserPage userPage = _resolveUserPage(userContent, siteName, sitemapName);
            _saxUserPage(contentHandler, userPage, logger, args);
        }
        
        /**
         * Gets the search context to use to determine site and sitemap as to resolve user pages from user contents
         * @param args The search arguments
         * @return The search context
         */
        protected SearchContext _getSearchContext(SearchComponentArguments args)
        {
            // If several contexts are selected with different sites, the behavior would probably not be the expected one
            // Only the pages of the first site would be SAXed
            // As a result we do not allow this configuration
            // The practical case is to select only one site, so this is not a restriction too much strict
            Collection<SearchContext> searchContexts = args.serviceInstance().getContexts();
            if (searchContexts.size() != 1)
            {
                throw new IllegalStateException("A search context is mandatory with UserReturnable in order to resolve the corresponding user pages. Multiple contexts are not allowed too.");
            }
            return searchContexts.iterator().next();
        }
        
        /**
         * Gets the site name
         * @param args The search arguments
         * @param searchContext The search context
         * @return the site name
         */
        protected String _getSiteName(SearchComponentArguments args, SearchContext searchContext)
        {
            SiteContext siteContext = searchContext.siteContext();
            switch (siteContext.getType())
            {
                case CURRENT:
                    return args.currentSite().getName();
                case AMONG:
                    List<Site> sites = siteContext.getSites().get();
                    if (sites.size() == 1)
                    {
                        return sites.iterator().next().getName();
                    }
                    throw new IllegalStateException("Site context must select one and only one site.");
                case ALL:
                case OTHERS:
                default:
                    throw new IllegalStateException("Site context must select one and only one site.");
            }
        }
        
        /**
         * Gets the sitemap name
         * @param args The search arguments
         * @param searchContext The search context
         * @param userContent The user content (current hit)
         * @return the sitemap name
         */
        protected String _getSitemapName(SearchComponentArguments args, SearchContext searchContext, Content userContent)
        {
            SitemapContext sitemapContext = searchContext.sitemapContext();
            switch (sitemapContext.getType())
            {
                case CURRENT_SITE:
                    String contentLanguage = userContent.getLanguage();
                    return contentLanguage == null
                            ? args.currentPage().getSitemapName()
                            : contentLanguage;
                case CHILD_PAGES:
                case DIRECT_CHILD_PAGES:
                    return args.currentPage().getSitemapName();
                case CHILD_PAGES_OF:
                case DIRECT_CHILD_PAGES_OF:
                    List<Page> pages = sitemapContext.getPages().get();
                    long nbSites = pages.stream()
                            .map(Page::getSite)
                            .distinct()
                            .count();
                    long nbSitemaps = pages.stream()
                            .map(Page::getSitemap)
                            .distinct()
                            .count();
                    if (nbSites != 1 || nbSitemaps != 1)
                    {
                        throw new IllegalStateException("Sitemap context must select one and only one sitemap.");
                    }
                    return pages.iterator().next().getSitemapName();
                default:
                    throw new IllegalStateException("Sitemap context must select one and only one sitemap.");
            }
        }
        
        /**
         * Resolves the use page corresponding to the user content
         * @param userContent The user content
         * @param siteName The site name
         * @param sitemapName The sitemap name
         * @return The user page
         */
        protected UserPage _resolveUserPage(Content userContent, String siteName, String sitemapName)
        {
            for (String contentType : _allContentTypes)
            {
                UserPage userPage = _userDirectoryPageResolver.getUserPage(userContent, siteName, sitemapName, contentType);
                if (userPage != null)
                {
                    return userPage;
                }
            }
            
            String msg = String.format("The page associated to the user content %s cannot be found in the current site and sitemap (%s / %s) for one of the content types %s", 
                    userContent, 
                    siteName,
                    sitemapName,
                    _allContentTypes);
            throw new IllegalArgumentException(msg);
        }
        
        /**
         * Sax the user page
         * @param contentHandler The content handler
         * @param userPage The user page
         * @param logger The logger
         * @param args The search arguments
         * @throws SAXException if a SAX error occured
         */
        protected void _saxUserPage(ContentHandler contentHandler, UserPage userPage, Logger logger, SearchComponentArguments args) throws SAXException
        {
            super.sax(contentHandler, userPage, logger, args);
        }
    }
}
