/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.synchronize;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.slf4j.Logger;

import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.contentio.synchronize.impl.SQLSynchronizableContentsCollection;
import org.ametys.plugins.userdirectory.DeleteOrgUnitComponent;
import org.ametys.plugins.userdirectory.DeleteUserComponent;

/**
 * SQL synchronizable collection for users
 */
public class SQLSynchronizableUsersCollection extends SQLSynchronizableContentsCollection
{
    private static final String __PARAM_SQL_TABLE_LOGIN = "login";
    private static final String __PARAM_POPULATION_ID = "populationId";
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The delete user component */
    protected DeleteUserComponent _deleteUserComponent;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _deleteUserComponent = (DeleteUserComponent) smanager.lookup(DeleteUserComponent.ROLE);
    }
    
    public boolean handleRightAssignmentContext()
    {
        return false;
    }
    
    @Override
    public String getIdField()
    {
        return UserSCCConstants.USER_UNIQUE_ID_ATTRIBUTE_NAME;
    }
    
    @Override
    public Map<String, List<String>> getMapping()
    {
        Map<String, List<String>> mapping = super.getMapping();
        mapping.put(getIdField(), Collections.singletonList(getLoginSQLColumnName()));
        
        return mapping;
    }
    
    /**
     * Get the login column name of user SQL table
     * @return The login column name of user SQL table
     */
    public String getLoginSQLColumnName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_TABLE_LOGIN);
    }
    
    /**
     * Get population Id
     * @return The population Id
     */
    public String getPopulationId()
    {
        return (String) getParameterValues().get(__PARAM_POPULATION_ID);
    }
    
    @Override
    protected Map<String, Object> _getSearchParameters(Map<String, Object> parameters, int offset, int limit, List<Object> sort, List<String> columns)
    {
        // Add the sql column name for the login.
        String loginSQLColumn = getLoginSQLColumnName();
        if (!columns.contains(loginSQLColumn))
        {
            columns.add(loginSQLColumn);
        }
        
        return super._getSearchParameters(parameters, offset, limit, sort, columns);
    }
    
    @Override
    protected boolean _checkIdObjectValue(String remoteKey, Object idObjectValue, Logger logger)
    {
        if (!super._checkIdObjectValue(remoteKey, idObjectValue, logger))
        {
            return false;
        }
        
        String login = idObjectValue.toString();
        String population = getPopulationId();
        if (_userManager.getUser(population, login) == null)
        {
            logger.warn("The user " + login + " don't belong to population " + population);
            return false;
        }
        
        return true;
    }
    
    @Override
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        Map<String, Object> additionalRemoteValues = super.getAdditionalAttributeValues(idValue, content, additionalParameters, create, logger);
        UserIdentity user = new UserIdentity(idValue, getPopulationId());
        additionalRemoteValues.put(UserSCCConstants.USER_ATTRIBUTE_NAME, user);
        return additionalRemoteValues;
    }
    
    @Override
    protected int _deleteContents(List<Content> contentsToRemove, Logger logger)
    {
        return _deleteUserComponent.deleteContentsWithLog(contentsToRemove, Map.of(DeleteOrgUnitComponent.SCC_ID_PARAMETERS_KEY, getId()), Map.of(), logger);
    }
}
