/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Seek any user content with email starting by the error string [Ljava.lang.String;@
// To reset it to the email associated in the user from the database

const ContentTypesHelper = Ametys.serviceManager.lookup(org.ametys.cms.contenttype.ContentTypesHelper.ROLE);
const UserManager = Ametys.serviceManager.lookup(org.ametys.core.user.UserManager.ROLE);

const contentTypes = ContentTypesHelper.getContentTypesList(java.util.List.of("org.ametys.plugins.userdirectory.Content.user"), true, false, true, false, false)
    .get("contentTypes")
    .stream()
    .map(contentType => contentType.get("id"))
    .filter(id => { try { return ContentTypesHelper.getModelItem("email", [id], []) != null; } catch (e) { return false; } })
    .collect(java.util.stream.Collectors.toList());

if (contentTypes.length == 0)
{
    logger.info("No content type is concerned. Skipping");
    return;
}

const predicate = contentTypes.stream()
                              .map(id => `@ametys-internal:contentType='${id}'`)
                              .collect(java.util.stream.Collectors.joining(" or "));
const query = `//element(*, ametys:content)[(${predicate}) and jcr:like(@ametys:email, '[Ljava.lang.String;@%')]`;

const contents = Repository.query(query);

let totalContents = contents.getSize();
let handledContents = 0;
logger.info(`${totalContents} user contents with email issues need to be fixed`);

contents.forEach(
    content =>
    {
        try
        {
            const userIdentity = content.getValue("user", false, null);
            if (userIdentity != null)
            {
                const user = UserManager.getUser(userIdentity);
                if (user != null)
                {
                    const email = user.getEmail()
                    if (email)
                    {
                        logger.debug(`Content ${content.getTitle()} fixed with email ${email}.`);
                        Content.migrate(content, [_fixEmail.bind(null, email)], false, null, false, true);
                    }
                    else
                    {
                        logger.warn(`Content ${content.getTitle()} (${content.getId()}) is related to a user of id ${org.ametys.core.user.UserIdentity.userIdentityToString(userIdentity)} with no email and thus cannot be fixed`);
                    }
                }
                else
                {
                    logger.warn(`Content ${content.getTitle()} (${content.getId()}) is related to the unknwow user of id ${org.ametys.core.user.UserIdentity.userIdentityToString(userIdentity)} and thus cannot be fixed`);
                }
            }
            else
            {
                logger.warn(`Content ${content.getTitle()} (${content.getId()}) has no related user... strange`);
            }
        }
        catch (e)
        {
            logger.error(`Content ${content.getId()} migration failure`, e);
            throw e;
        }
    
        handledContents++;
        if (handledContents % 500 == 0)
        {
            logger.info(`[${handledContents}/${totalContents}] handled contents`);
        }
    }
);

logger.info(`${totalContents} user contents handled`);

function _fixEmail(email, content)
{
    content.setValue("email", email);
}
