/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Delete user content that belongs to an unknown user after a delay
 */
public class UserContentDataPolicy extends AbstractLogEnabled implements PersonalDataPolicy, Serviceable
{
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    /** The user content delete component */
    protected DeleteUserComponent _userContentDAO;
    private Period _retentionPeriod;

    public void service(ServiceManager manager) throws ServiceException
    {
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
        _userContentDAO = (DeleteUserComponent) manager.lookup(DeleteUserComponent.ROLE);
        
        Long config = Config.getInstance().<Long>getValue("user-directory.user.content.data-policy.retention.period", false, null);
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo)
    {
        if (_retentionPeriod == null)
        {
            return AnonymizationResult.TOO_EARLY;
        }
        else if (userStatusInfo.getMissingSinceDate().isBefore(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            UserIdentity userIdentity = userStatusInfo.getUserIdentity();
            List<Content> userContents = _userDirectoryHelper.getUserContents(userIdentity);
            if (userContents.size() > 0)
            {
                _userContentDAO.deleteContents(userContents.stream().map(Content::getId).toList(), Map.of(), Map.of(), getLogger());
                ForensicLogger.info("data.policy.gdpr.remove.user.contents", Map.of("handled", Integer.toString(userContents.size()), "identity", userIdentity), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                return AnonymizationResult.PROCESSED;
            }
            return AnonymizationResult.NO_DATA;
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }

}
