/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.observation;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.ObservationConstants;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.userdirectory.DeleteUserComponent;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Delete user contents of deleted user
 */
public class UserDeletedObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    /** The delete user component */
    protected DeleteUserComponent _deleteUserComponent;

    public void service(ServiceManager manager) throws ServiceException
    {
        _deleteUserComponent = (DeleteUserComponent) manager.lookup(DeleteUserComponent.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
    }
    
    public int getPriority()
    {
        return Integer.MAX_VALUE;
    }

    public boolean supports(Event event)
    {
        return ObservationConstants.EVENT_USER_DELETED.equals(event.getId());
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        UserIdentity identity = (UserIdentity) event.getArguments().get(ObservationConstants.ARGS_USER);
        List<Content> userContents = _userDirectoryHelper.getUserContents(identity);
        
        int deletedContents = _deleteUserComponent.deleteContentsWithLog(userContents, Map.of(), Map.of(), getLogger());
        if (userContents.size() != deletedContents)
        {
            getLogger().error("Something prevented the user content of a deleted user to be deleted");
        }
    }
}
