/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.forms;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditionsHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.disableconditions.DisableConditions.ASSOCIATION_TYPE;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Helper for adding UD attributes to form questions
 */
public class FormQuestionUDHelper implements Serviceable, Component
{
    /** The avalon role. */
    public static final String ROLE = FormQuestionUDHelper.class.getName();
    
    /** The content type attribute */
    public static final String ATTRIBUTE_USER_CONTENT_TYPE = "user-content-type";
    
    /** The user info attribute */
    public static final String ATTRIBUTE_UD_USER_ATTRIBUTE_NAME = "ud-user-info";
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The user directory helper */
    protected UserDirectoryHelper _udHelper;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The data holder relative disable condition helper */
    protected DataHolderRelativeDisableConditionsHelper _disableConditionsHelper;

    /** The form element definition helper */
    protected FormElementDefinitionHelper _formElementDefinitionHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _udHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _disableConditionsHelper = (DataHolderRelativeDisableConditionsHelper) manager.lookup(DataHolderRelativeDisableConditionsHelper.ROLE);
        _formElementDefinitionHelper = (FormElementDefinitionHelper) manager.lookup(FormElementDefinitionHelper.ROLE);
    }
    
    /**
     * Get user value
     * @param question the question
     * @param user the user
     * @return the user value.<code>null</code> if there is not user value
     */
    public Object getUserContentAttributeValue(FormQuestion question, UserIdentity user)
    {
        String contentTypeId = question.getValue(ATTRIBUTE_USER_CONTENT_TYPE);
         
        String userAttribute = question.getValue(ATTRIBUTE_UD_USER_ATTRIBUTE_NAME + "-" + contentTypeId.replace(".", "-"));
        List<Content> userContents = _udHelper.getUserContents(user, _formDAO.getFormLocale(question.getForm()));
        Optional<Content> userContent = userContents.stream().filter(u -> ArrayUtils.contains(u.getTypes(), contentTypeId)).findAny();
        return (userContent.isPresent()) ? userContent.get().getValue(userAttribute) : null;
    }
    
    /**
     * Get the list of model items for selecting a user directory attribute
     * @param acceptedTypes accepted stockage type of the attributes
     * @return the list of model items
     */
    public Map<String, ModelItem> getUDModelItems(List<String> acceptedTypes)
    {
        Map<String, ModelItem> modelItems = new LinkedHashMap<>();
        ElementDefinition<String> contentTypes = _formElementDefinitionHelper.getElementDefinition(FormQuestionUDHelper.ATTRIBUTE_USER_CONTENT_TYPE, ModelItemTypeConstants.STRING_TYPE_ID, "plugin.user-directory:PLUGINS_USER_DIRECTORY_QUESTIONS_DIALOG_COMPUTING_CONTENT_TYPE", "plugin.user-directory:PLUGINS_USER_DIRECTORY_QUESTIONS_DIALOG_COMPUTING_CONTENT_TYPE_DESC", new DefaultValidator(null, true));
        StaticEnumerator<String> contentTypesStaticEnumerator = new StaticEnumerator<>();
        contentTypes.setEnumerator(contentTypesStaticEnumerator);
        modelItems.put(contentTypes.getName(), contentTypes);
        
        for (String ctId : _contentTypeEP.getSubTypes(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE))
        {
            ContentType contentType = _contentTypeEP.getExtension(ctId);
            contentTypesStaticEnumerator.add(contentType.getLabel(), ctId);

            ElementDefinition<String> contentAttributes = _getContentAttributesModelItem(acceptedTypes, ctId, contentType);
            modelItems.put(contentAttributes.getName(), contentAttributes);
        }
        return modelItems;
    }

    private ElementDefinition<String> _getContentAttributesModelItem(List<String> acceptedTypes, String ctId, ContentType contentType)
    {
        ElementDefinition<String> contentAttributes = _formElementDefinitionHelper.getElementDefinition(FormQuestionUDHelper.ATTRIBUTE_UD_USER_ATTRIBUTE_NAME + "-" + ctId.replace(".", "-"), ModelItemTypeConstants.STRING_TYPE_ID, "plugin.user-directory:PLUGINS_USER_DIRECTORY_QUESTIONS_DIALOG_COMPUTING_UD_USERS", "plugin.user-directory:PLUGINS_USER_DIRECTORY_QUESTIONS_DIALOG_COMPUTING_UD_USERS_DESC", new DefaultValidator(null, true));
        StaticEnumerator<String> userInfoStaticEnumerator = _getContentTypeAttributesEnumerator(contentType, acceptedTypes);
        
        contentAttributes.setDisableConditions(_getUDAttributeDisableCondition(ctId));
        contentAttributes.setEnumerator(userInfoStaticEnumerator);
        return contentAttributes;
    }

    private DisableConditions _getUDAttributeDisableCondition(String ctId)
    {
        DisableConditions disableConditions = new DataHolderRelativeDisableConditions();
        DisableCondition condition = new DataHolderRelativeDisableCondition(FormQuestionUDHelper.ATTRIBUTE_USER_CONTENT_TYPE, OPERATOR.NEQ, ctId, _disableConditionsHelper);
        disableConditions.getConditions().add(condition);
        disableConditions.setAssociation(ASSOCIATION_TYPE.OR);
        return disableConditions;
    }
    
    private StaticEnumerator<String> _getContentTypeAttributesEnumerator(ContentType contentType, List<String> acceptedTypes)
    {
        StaticEnumerator<String> userInfoStaticEnumerator = new StaticEnumerator<>();
        for (ModelItem modelItem : contentType.getModelItems())
        {
            if (acceptedTypes.contains(modelItem.getType().getId()) && modelItem instanceof ElementDefinition elmtDef)
            {
                if (!elmtDef.isMultiple())
                {
                    userInfoStaticEnumerator.add(modelItem.getLabel(), modelItem.getPath());
                }
            }
        }
        return userInfoStaticEnumerator;
    }
}
