/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.generator;

import java.io.IOException;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.web.content.ContentGenerator;

/**
 * OrgUnit Generator
 */
public class OrgUnitGenerator extends ContentGenerator
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The organization chart page handler */
    protected OrganisationChartPageHandler _organizationChartPageHandler;

    /** The content helper */
    protected ContentHelper _contentHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _srcResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _organizationChartPageHandler = (OrganisationChartPageHandler) serviceManager.lookup(OrganisationChartPageHandler.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
    }
    
    @Override
    protected void _saxOtherData(Content content) throws SAXException, ProcessingException
    {
        super._saxOtherData(content);
        
        String viewName = parameters.getParameter("viewName", "main");
        if ("main".equals(viewName))
        {
            // Sax users and child orgunits only for view "main"
            _saxUsers(content);
            _saxChildOrgUnits(content);
        }
    }
    
    /**
     * Sax child orgUnits from the orgUnit content
     * @param content the content
     * @throws SAXException if an error occurred
     */
    protected void _saxChildOrgUnits(Content content) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "orgUnits");
        for (Content childOrgUnit : _organizationChartPageHandler.getChildContents(content))
        {
            saxOrgUnit(childOrgUnit, "link");
        }
        XMLUtils.endElement(contentHandler, "orgUnits");
    }
    
    /**
     * SAX a orgUnit content
     * @param orgUnit the orgUnit to sax.
     * @param viewName the view name
     * @throws SAXException if an error occurs
     */
    protected void saxOrgUnit(Content orgUnit, String viewName) throws SAXException
    {
        try
        {
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("metadataSetName", viewName);
            XMLUtils.startElement(contentHandler, "orgUnit", atts);
            saxContent(orgUnit, viewName);
            XMLUtils.endElement(contentHandler, "orgUnit");
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
    }
    
    /**
     * Sax users from the orgUnit content
     * @param content the content
     * @throws SAXException if an error occurred
     */
    protected void _saxUsers(Content content) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "users");
        
        if (content.hasValue("users"))
        {
            ModelAwareRepeater users = content.getRepeater("users");
            for (ModelAwareRepeaterEntry entry: users.getEntries())
            {
                if (entry.hasValue("user"))
                {
                    ContentValue userValue = entry.getValue("user");
                    Optional<? extends Content> optionalUser = userValue.getContentIfExists();
                    
                    if (optionalUser.isPresent())
                    {
                        Content user = optionalUser.get();
                        String role = entry.getValue("role");
                        saxUser(user, role, "abstract");
                        saxUser(user, role, "link");
                    }
                    else
                    {
                        getLogger().error("Can't find the user from id : " + userValue.getContentId());
                    }
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, "users");
    }
    
    /**
     * SAX a user content
     * @param user the user.
     * @param role the user role
     * @param viewName the view name
     * @throws SAXException if an error occurs
     */
    protected void saxUser(Content user, String role, String viewName) throws SAXException
    {
        try
        {
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("metadataSetName", viewName);
            if (StringUtils.isNotBlank(role))
            {
                atts.addCDATAAttribute("role", role);
            }

            XMLUtils.startElement(contentHandler, "user", atts);
            saxContent(user, viewName);
            XMLUtils.endElement(contentHandler, "user");
        }
        catch (IOException e)
        {
            throw new SAXException(e);
        }
    }
    
    /**
     * SAX the HTML content of a {@link Content}
     * @param content the content
     * @param viewName the view name
     * @throws SAXException If an error occurred saxing the content
     * @throws IOException If an error occurred resolving the content
     */
    protected void saxContent (Content content, String viewName) throws SAXException, IOException
    {
        String format = parameters.getParameter("output-format", "html");
        if (StringUtils.isEmpty(format))
        {
            format = "html";
        }
        
        saxContent(content, viewName, format);
    }
    
    /**
     * SAX a {@link Content} to given format
     * @param content the content
     * @param viewName the view name
     * @param format the output format
     * @throws SAXException If an error occurred saxing the content
     * @throws IOException If an error occurred resolving the content
     */
    protected void saxContent (Content content, String viewName, String format) throws SAXException, IOException
    {
        SitemapSource src = null;      
        try
        {
            String uri = _contentHelper.getContentViewUrl(content, viewName, format);
            src = (SitemapSource) _srcResolver.resolveURI(uri);
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", content.getId());
            attrs.addCDATAAttribute("name", content.getName());
            attrs.addCDATAAttribute("title", content.getTitle());
            attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
            
            XMLUtils.startElement(contentHandler, "content", attrs);
            src.toSAX(new IgnoreRootHandler(contentHandler));
            XMLUtils.endElement(contentHandler, "content");
        }
        catch (UnknownAmetysObjectException e)
        {
            // The content may be archived
        }
        finally
        {
            _srcResolver.release(src);
        }
    }
}
