/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

/**
 * Invalidates the ZoneItem for user contents when the root page properties changed
 */
public class InvalidateZoneItemCacheOnRootUpdatedObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private UserDirectoryPageHandler _udPageHandler;
    private UserDirectoryPageResolver _udPageResolver;
    private PageElementCache _zoneItemCache;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _udPageHandler = (UserDirectoryPageHandler) smanager.lookup(UserDirectoryPageHandler.ROLE);
        _zoneItemCache = (PageElementCache) smanager.lookup(PageElementCache.ROLE + "/zoneItem");
        _udPageResolver = (UserDirectoryPageResolver) smanager.lookup(UserDirectoryPageResolver.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_USER_DIRECTORY_ROOT_UPDATED);
    }
    
    public int getPriority()
    {
        // processed just before front-office cache invalidation
        return MAX_PRIORITY + 3500;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Page rootPage = (Page) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_PAGE);
        boolean viewUpdated = (Boolean) event.getArguments().get(ObservationConstants.ARGS_USER_CONTENT_VIEW_UPDATED);
        
        if (viewUpdated)
        {
            // user content view has been changed, clear zone item cache
            AmetysObjectIterable<Content> userContents = _udPageHandler.getContentsForRootPage(rootPage);
            for (Content content : userContents)
            {
                _removeZoneItemCache(rootPage, content);
            }
        }
    }
    
    private void _removeZoneItemCache(Page rootPage, Content content)
    {
        UserPage userPage = _udPageResolver.getUserPage(rootPage, content);
        if (userPage != null)
        {
            AmetysObjectIterable< ? extends Zone> zones = userPage.getZones();
            for (Zone zone : zones)
            {
                for (ZoneItem zoneItem : zone.getZoneItems())
                {
                    if (zoneItem.getType().equals(ZoneType.CONTENT)) 
                    {
                        // Remove zone item cache for all workspaces
                        _zoneItemCache.removeItem(null, rootPage.getSiteName(), "CONTENT", zoneItem.getId());
                    }
                }
            }
        }
    }

}
