/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.contentio.synchronize.rights.SynchronizedRootContentHelper;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for user directory.
 */
public class UserDirectoryHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable, Contextualizable
{
    /** The avalon role. */
    public static final String ROLE = UserDirectoryHelper.class.getName();
    
    /** The root node name of the plugin */
    public static final String USER_DIRECTORY_ROOT_NODE = "userdirectory";
    
    /** The orgUnit parent attribute */
    public static final String ORGUNITS_ATTRIBUTE = "orgunits";
    
    /** The orgUnit content type */
    public static final String ORGUNIT_CONTENT_TYPE = "org.ametys.plugins.userdirectory.Content.udorgunit";
    
    /** The parent content type id */
    public static final String ABSTRACT_USER_CONTENT_TYPE = "org.ametys.plugins.userdirectory.Content.user";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    /** The languages manager */
    protected LanguagesManager _languagesManager;
    
    /** The context */
    protected Context _context;
    
    static class UserElementKey extends AbstractCacheKey
    {
        UserElementKey(String lang, UserIdentity userIdentity)
        {
            super(lang, userIdentity);
        }
        
        static UserElementKey of(String lang, UserIdentity userIdentity)
        {
            return new UserElementKey(lang, userIdentity);
        }
        
        static UserElementKey of(String lang)
        {
            return new UserElementKey(lang, null);
        }
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _languagesManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(ROLE,
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_USER_CACHE_LABEL"),
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_USER_CACHE_DESCRIPTION"),
                true,
                null);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Gets the root of user directory contents
     * @return the root of user directory contents
     */
    public AmetysObjectCollection getUserDirectoryRootContent()
    {
        return getUserDirectoryRootContent(false);
    }
    
    /**
     * Gets the root of user directory contents
     * @param create <code>true</code> to create automatically the root when missing.
     * @return the root of user directory contents
     */
    public AmetysObjectCollection getUserDirectoryRootContent(boolean create)
    {
        ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins/");
        
        boolean needSave = false;
        if (!pluginsNode.hasChild(USER_DIRECTORY_ROOT_NODE))
        {
            if (create)
            {
                pluginsNode.createChild(USER_DIRECTORY_ROOT_NODE, "ametys:unstructured");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/userdirectory' is missing");
            }
        }
        
        ModifiableTraversableAmetysObject userdirectoryNode = pluginsNode.getChild(USER_DIRECTORY_ROOT_NODE);
        if (!userdirectoryNode.hasChild(SynchronizedRootContentHelper.IMPORTED_CONTENTS_ROOT_NODE))
        {
            if (create)
            {
                userdirectoryNode.createChild(SynchronizedRootContentHelper.IMPORTED_CONTENTS_ROOT_NODE, "ametys:collection");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/userdirectory/ametys:contents' is missing");
            }
        }
        
        if (needSave)
        {
            pluginsNode.saveChanges();
        }
        
        return userdirectoryNode.getChild(SynchronizedRootContentHelper.IMPORTED_CONTENTS_ROOT_NODE);
    }
    
    /**
     * Get the list of orgunits content
     * @param user the user content
     * @return the list of orgunits
     */
    public List<Content> getOrgUnits(Content user)
    {
        ContentValue[] contents = user.getValue(ORGUNITS_ATTRIBUTE);
        if (contents == null)
        {
            return List.of();
        }
        
        return Arrays.stream(contents)
                     .map(v -> v.getContentIfExists())
                     .flatMap(Optional::stream)
                     .collect(Collectors.toList());
    }
    
    /**
     * Get all user content for the identity in all languages
     * @param user the identity
     * @return the user contents
     */
    public List<Content> getUserContents(UserIdentity user)
    {
        return getUserContents(user, null);
    }
    
    /**
     * Get user contents from user identity
     * @param user the user identity
     * @param givenLanguage the given language. if null, all contents from all available languages are returned
     * @return the user contents
     */
    public List<Content> getUserContents(UserIdentity user, String givenLanguage)
    {
        List<Content> userContents = new ArrayList<>();
        Iterator<String> languagesIterator = _getLanguagesToTest(givenLanguage).iterator();
        while (languagesIterator.hasNext()
            && (userContents.isEmpty() || givenLanguage == null))
        {
            String lang = languagesIterator.next();
            UserElementKey key = UserElementKey.of(lang, user);
            List<String> contentIds = _getCache().get(key, k -> _requestUserContentIds(user, lang));
            
            userContents.addAll(contentIds.stream()
                    .map(id -> this._resolveContent(id, lang))
                    .filter(Objects::nonNull)
                    .toList());
        }
        
        return userContents;
    }
    
    private Content _resolveContent(String contentId, String lang)
    {
        try
        {
            return contentId != null ? _resolver.resolveById(contentId) : null;
        }
        catch (UnknownAmetysObjectException e)
        {
            // The content's id is always retrieved in default workspace: content may not exist in current workspace
            getLogger().warn("User content with id '{}' was not found in current workspace for language '{}'", contentId, lang);
        }
        
        return null;
    }
    
    /**
     * Get user content from user identity
     * @param user the user identity
     * @param givenLanguage the given language
     * @return the user content (null if it not exist)
     */
    public Content getUserContent(UserIdentity user, String givenLanguage)
    {
        List<Content> userContents = getUserContents(user, givenLanguage);
        return userContents.isEmpty() ? null : userContents.get(0);
    }
    
    private Set<String> _getLanguagesToTest(String givenLang)
    {
        Set<String> languages = new LinkedHashSet<>();
        
        // First, test the given language if not blank
        if (StringUtils.isNotBlank(givenLang))
        {
            languages.add(givenLang);
        }
        
        // Then test english language
        languages.add("en");
        
        // Finaly test other languages
        languages.addAll(_languagesManager.getAvailableLanguages().keySet());
        
        return languages;
    }

    /**
     * Request the list of user content id
     * @param user the user identity
     * @param lang the lang
     * @return the list of user content id (empty if it not exist)
     */
    protected List<String> _requestUserContentIds(UserIdentity user, String lang)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Use default workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
         
            // Content type condition
            List<String> cTypeIds = new ArrayList<>();
            cTypeIds.add(ABSTRACT_USER_CONTENT_TYPE);
            cTypeIds.addAll(_cTypeEP.getSubTypes(ABSTRACT_USER_CONTENT_TYPE));
            Expression cTyPeExpr = new ContentTypeExpression(Operator.EQ, cTypeIds.toArray(new String[cTypeIds.size()]));
            
            // Lang condition
            Expression langExpr = new LanguageExpression(Operator.EQ, lang);
            // Login condition
            Expression userExpr = new UserExpression("user", Operator.EQ, user);
            
            Expression finalExpr = new AndExpression(cTyPeExpr, langExpr, userExpr);
            String query = ContentQueryHelper.getContentXPathQuery(finalExpr);
            
            AmetysObjectIterable<Content> contents = _resolver.query(query);
            return contents.stream()
                    .map(Content::getId)
                    .toList();
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    private Cache<UserElementKey, List<String>> _getCache()
    {
        return _cacheManager.get(ROLE);
    }
    
    /**
     * Invalidate the user content cache for one language. If lang is <code>null</code>, invalidate all
     * @param lang the lang. Can be <code>null</code>
     */
    public void invalidateUserContentCache(String lang)
    {
        if (StringUtils.isNotBlank(lang))
        {
            _getCache().invalidate(UserElementKey.of(lang));
        }
        else
        {
            _getCache().invalidateAll();
        }
    }
}
