/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.clientsideelement.DeleteContentClientSideElement;
import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.userdirectory.DeleteUserComponent;

/**
 * This element creates an action button to delete a user content
 */
public class DeleteUserClientSideElement extends DeleteContentClientSideElement
{
    /** The delete orgUnit component */
    protected DeleteUserComponent _deleteUserComponent;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _deleteUserComponent = (DeleteUserComponent) smanager.lookup(DeleteUserComponent.ROLE);
    }
    
    @Override
    protected boolean _isContentReferenced(Content content)
    {
        return _deleteUserComponent.isContentReferenced(content, getLogger());
    }
    
    /**
     * Delete user contents
     * @param contentsId The ids of contents to delete
     * @param parameters the additional parameters
     * @return the deleted and undeleted contents
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteContents(List<String> contentsId, Map<String, Object> parameters)
    {
        Map<String, Object> results = new HashMap<>();
        
        Map<String, Map<String, Object>> initialContentParameters = new HashMap<>();
        for (String contentId : contentsId)
        {
            initialContentParameters.put(contentId, getContentDefaultParameters(_resolver.resolveById(contentId)));
        }
        
        Map<String, Object> deleteResults = _deleteUserComponent.trashContents(contentsId, parameters, _rights, getLogger());
        for (String contentId : contentsId)
        {
            Map<String, Object> result = new HashMap<>();
            Map<String, Object> deleteResult = (Map<String, Object>) deleteResults.get(contentId);
            if (deleteResult.containsKey("check-before-deletion-failed"))
            {
                result.put("check-before-deletion-failed", deleteResult.get("check-before-deletion-failed"));
            }
            
            String initialContentId = (String) deleteResult.get("initial-content");
            result.put("initial-content", initialContentParameters.get(initialContentId));
            
            List<Map<String, Object>> deletedContents = new ArrayList<>();
            for (String deleteContentId : (List<String>) deleteResult.get("deleted-contents"))
            {
                Map<String, Object> deleteParameters = new HashMap<>();
                deleteParameters.put("id", deleteContentId);
                deletedContents.add(deleteParameters);
            }
            result.put("deleted-contents", deletedContents);
            
            List<Map<String, Object>> undeletedContents = new ArrayList<>();
            for (Content content : (List<Content>) deleteResult.get("undeleted-contents"))
            {
                undeletedContents.add(getContentDefaultParameters(content));
            }
            result.put("undeleted-contents", undeletedContents);
            
            List<Map<String, Object>> referencedContents = new ArrayList<>();
            for (Content content : (List<Content>) deleteResult.get("referenced-contents"))
            {
                referencedContents.add(getContentDefaultParameters(content));
            }
            result.put("referenced-contents", referencedContents);
            
            List<Map<String, Object>> unauthorizedContents = new ArrayList<>();
            for (Content content : (List<Content>) deleteResult.get("unauthorized-contents"))
            {
                Map<String, Object> contentDefaultParameters = getContentDefaultParameters(content);
                contentDefaultParameters.put("description", _getNoRightDescription(content));
                unauthorizedContents.add(contentDefaultParameters);
            
            }
            result.put("unauthorized-contents", unauthorizedContents);
            
            List<Map<String, Object>> lockedContents = new ArrayList<>();
            for (Content content : (List<Content>) deleteResult.get("locked-contents"))
            {
                Map<String, Object> contentDefaultParameters = getContentDefaultParameters(content);
                contentDefaultParameters.put("description", _getLockedDescription(content));
                lockedContents.add(contentDefaultParameters);
            }
            result.put("locked-contents", lockedContents);
            
            results.put(contentId, result);
        }
        
        return results;
    }
}
