/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.web.repository.page.Page;

/**
 * Resolves an organisation chart page path from the associated orgUnit content.
 */
public class OrganisationChartPageResolver extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role. */
    public static final String ROLE = OrganisationChartPageResolver.class.getName();
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    /** The organisation chart page handler */
    protected OrganisationChartPageHandler _pageHandler;
    
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _pageHandler = (OrganisationChartPageHandler) serviceManager.lookup(OrganisationChartPageHandler.ROLE);
    }
    
    /**
     * Return the orgUnit page
     * @param orgUnitRootPage the orgUnit root page
     * @param orgUnitContent the orgUnit content
     * @return the orgUnit page or null
     */
    public OrgUnitPage getOrgUnitPage(Page orgUnitRootPage, Content orgUnitContent)
    {
        // E.g: udorgunit://path?rootId=...&contentId=...
        String orgUnitPath = _getOrgUnitPath(orgUnitContent);
        String pageId =  "udorgunit://" + orgUnitPath + "?rootId=" + orgUnitRootPage.getId() + "&contentId=" + orgUnitContent.getId();
        return _ametysResolver.resolveById(pageId);
    }
    
    /**
     * Get the path of a orgUnit content
     * @param orgUnitContent The orgUnit content
     * @return The path
     */
    private String _getOrgUnitPath(Content orgUnitContent)
    {
        Content parentContent = _pageHandler.getParentContent(orgUnitContent);
        if (parentContent != null)
        {
            return _getOrgUnitPath(parentContent) + "/" + orgUnitContent.getName();
        }
        else
        {
            return orgUnitContent.getName();
        }
    }

    /**
     * Return the orgUnit page
     * @param userContent the orgUnit page content
     * @param siteName The current site name.
     * @param sitemapName The current sitemap name.
     * @return the orgUnit page or null or <code>null</code> if the orgUnit page does not exist
     */
    public OrgUnitPage getOrgUnitPage(Content userContent, String siteName, String sitemapName)
    {
        for (String contentTypeId : userContent.getTypes())
        {
            OrgUnitPage orgUnitPage = getOrgUnitPage(userContent, siteName, sitemapName, contentTypeId);
            if (orgUnitPage != null)
            {
                return orgUnitPage;
            }
        }
        
        return null;
    }
    
    /**
     * Return the orgUnit page
     * @param userContent the orgUnit page content
     * @param siteName The current site name.
     * @param sitemapName The current sitemap name.
     * @param contentTypeId the content type id
     * @return the orgUnit page or null
     */
    public OrgUnitPage getOrgUnitPage(Content userContent, String siteName, String sitemapName, String contentTypeId)
    {
        String language = userContent.getLanguage();
        if (language == null)
        {
            language = sitemapName;
        }
        
        Page organisationChartRootPage = _pageHandler.getOrganisationChartRootPage(siteName, language, contentTypeId);
        
        if (organisationChartRootPage == null)
        {
            return null;
        }
        
        return getOrgUnitPage(organisationChartRootPage, userContent);
    }
}
