/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.AbstractConfigurableVirtualPage;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Page representing an orgUnit page.
 */
public class OrgUnitPage extends AbstractConfigurableVirtualPage<OrgUnitPageFactory>
{
    private String _title;
    private String _path;
    private Content _syncContent;
    
    /**
     * Constructor.
     * @param root the root page.
     * @param syncContent the synchronized content
     * @param path the path
     * @param configuration The abstract virtual page configuration
     * @param scheme The scheme
     * @param factory The factory
     */
    public OrgUnitPage(Page root, VirtualPageConfiguration configuration, String scheme, OrgUnitPageFactory factory, Content syncContent, String path)
    {
        super(root, configuration, scheme, factory);

        _path = path;
        _syncContent = syncContent;
        
        _title = _syncContent.getTitle(LocaleUtils.toLocale(root.getSitemapName()));
    }
    
    /**
     * Returns the associated synchronizable {@link Content}.
     * @return the associated synchronizable {@link Content}.
     */
    @SuppressWarnings("unchecked")
    @Override
    public Content getContent()
    {
        return _syncContent;
    }
    
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + (_path != null ? _path.split("/").length : 0);
    }
    
    @Override
    public Set<String> getReferers() throws AmetysRepositoryException
    {
        return null;
    }

    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _title;
    }

    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        
        for (Content content : _factory.getOrganisationChartPageHandler().getChildContents(_syncContent))
        {
            children.add(_factory.createOrgUnitPage(_root, content, _path + "/" + content.getName()));
        }
        
        return new CollectionIterable<>(children);
    }

    
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        return _root.getPathInSitemap() + "/" + _path;
    }

    @SuppressWarnings("unchecked")
    
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        Content childContent = _factory.getOrganisationChartPageHandler().getChildFromPath(_syncContent, path);
        if (childContent != null)
        {
            OrgUnitPage page = _factory.createOrgUnitPage(_root, childContent, _path + "/" + path);
            return (A) page;
        }
        
        return null;
    }
    

    @SuppressWarnings("unchecked")
    
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return !_factory.getOrganisationChartPageHandler().getChildContents(_syncContent).stream().filter(c -> c.getName().equals(name)).collect(Collectors.toList()).isEmpty();
    }
    
    
    public String getId() throws AmetysRepositoryException
    {
        // E.g: udorgunit://path?rootId=...&contentId=...
        return "udorgunit://" + _path + "?rootId=" + _root.getId() + "&contentId=" + _syncContent.getId();
    }

    
    public String getName() throws AmetysRepositoryException
    {
        return _syncContent.getName();
    }
    
    @SuppressWarnings("unchecked")
    
    public Page getParent() throws AmetysRepositoryException
    {
        Content parentContent = _factory.getOrganisationChartPageHandler().getParentContent(_syncContent);
        if (parentContent != null)
        {
            return _factory.createOrgUnitPage(_root, parentContent, StringUtils.substringBeforeLast(_path, "/"));
        }

        return _root;
    }

    
    public String getParentPath() throws AmetysRepositoryException
    {
        if (_path.contains("/"))
        {
            return _root.getPath() + "/" + StringUtils.substringBeforeLast(_path, "/");
        }
        else
        {
            return _root.getPath();
        }
    }

    @Override
    public String getPath() throws AmetysRepositoryException
    {
        return _root.getPath() + "/" + _path;
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePage) throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        throw new UnknownAmetysObjectException("There is no child for orgUnit page");
    }
    
    @Override
    public boolean isVisible() throws AmetysRepositoryException
    {
        return _root.getValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME, true);
    }
}
