/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.rights;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a User directory {@link Content}
 */
public class UserDirectoryContentAccessController extends AbstractHierarchicalAccessController<AmetysObject> implements Contextualizable
{
    /** the user directory context category */
    public static final I18nizableText USER_DIRECTORY_CONTEXT_CATEGORY = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_RIGHT_ASSIGNMENT_CONTEXT_CONTENTS_LABEL");

    /** Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The organisation chart page handler */
    protected OrganisationChartPageHandler _organisationChartPageHandler;
    
    /** The content type helper */
    protected ContentTypesHelper _contentTypeHelper;
    
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;

    private Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _organisationChartPageHandler = (OrganisationChartPageHandler) manager.lookup(OrganisationChartPageHandler.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public boolean supports(Object object)
    {
        if (object instanceof Content)
        {
            return _isContentUser(object) || _isContentOrgUnit(object);
        }
        
        return false;
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        if (object instanceof Content)
        {
            Set<AmetysObject> parents = new HashSet<>();
            
            Request request = ContextHelper.getRequest(_context);
            String originalWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
            try
            {
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
                if (_isContentOrgUnit(object))
                {
                    Content parentContent = _organisationChartPageHandler.getParentContent((Content) object);
                    if (parentContent != null)
                    {
                        parents.add(parentContent);
                    }
                    else
                    {
                        parents.add(_userDirectoryHelper.getUserDirectoryRootContent());
                    }
                }
                else if (_isContentUser(object))
                {
                    parents.add(_userDirectoryHelper.getUserDirectoryRootContent());
                }
            }
            finally
            {
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, originalWorkspace);
            }
            
            return parents;
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            try
            {
                return Collections.singleton(_userDirectoryHelper.getUserDirectoryRootContent());
            }
            catch (UnknownAmetysObjectException e)
            {
                // Root does not exist yet
                return null;
            }
        }
        return null;
    }
    
    private boolean _isContentUser(Object object)
    {
        return _contentTypeHelper.isInstanceOf((Content) object, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
    }
    
    private boolean _isContentOrgUnit(Object object)
    {
        return _contentTypeHelper.isInstanceOf((Content) object, UserDirectoryHelper.ORGUNIT_CONTENT_TYPE);
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Content content)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            if (_isContentUser(content))
            {
                return new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CONTENT_ACCESS_CONTROLLER_USER_CONTEXT_EXPLANATION_LABEL", params);
            }
            else if (_isContentOrgUnit(content))
            {
                return new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CONTENT_ACCESS_CONTROLLER_ORGUNIT_CONTEXT_EXPLANATION_LABEL", params);
            }
        }
        else if (_userDirectoryHelper.getUserDirectoryRootContent().equals(object))
        {
            return new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CONTENT_ACCESS_CONTROLLER_ROOT_CONTEXT_EXPLANATION_LABEL");
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        if (object instanceof Content content)
        {
            return new I18nizableText(content.getTitle());
        }
        else if (_userDirectoryHelper.getUserDirectoryRootContent().equals(object))
        {
            return new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CONTENT_ACCESS_CONTROLLER_ROOT_CONTEXT_LABEL");
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return USER_DIRECTORY_CONTEXT_CATEGORY;
    }
    
    @Override
    public int getObjectPriority(Object object)
    {
        if (_userDirectoryHelper.getUserDirectoryRootContent().equals(object))
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }
}
