/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.signup;

import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.user.User;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.usermanagement.UserManagementException;

import com.google.common.collect.Multimap;
import com.opensymphony.workflow.WorkflowException;

/**
 * This component overrides {@link org.ametys.web.usermanagement.UserSignupManager} to add the creation of a user content on signup.
 */
public class UserSignupManager extends org.ametys.web.usermanagement.UserSignupManager
{
    /** The default workflow action id for initializing user content */
    protected int _initWorkflowActionId;
    /** The default workflow name user content */
    protected String _workflowName;
    
    private Context _context;
    private ContentTypeExtensionPoint _contentTypeEP;
    private FOContentCreationHelper _foHelper;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        super.contextualize(context);
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _contentTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _foHelper = (FOContentCreationHelper) serviceManager.lookup(FOContentCreationHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _initWorkflowActionId = Integer.valueOf(configuration.getChild("user-init-action-id").getValue());
        _workflowName = configuration.getChild("user-workflow-name").getValue();
    }
    
    /**
     * Get the request
     * @return the request
     */
    protected Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
    
    @Override
    public void validationBeforeSignup(Multimap<String, I18nizableText> errors)
    {
        Request request = _getRequest();
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        if (zoneItem != null)
        {
            String userContentTypeId = zoneItem.getServiceParameters().getValue("user-content-type");
            if (StringUtils.isNotBlank(userContentTypeId))
            {
                ContentType userContentType = _contentTypeEP.getExtension(userContentTypeId);
                if (userContentType != null)
                {
                    Map<String, Object> userValues = _foHelper.getAndValidateFormValues(request, userContentType, "signup", errors);
                    if (errors.isEmpty())
                    {
                        request.setAttribute(UserSignupManager.class.getName() + "$userValues", userValues);
                    }
                }
            }
        }
    }
    
    @Override
    public void additionalSignupOperations(User createdUser, Multimap<String, I18nizableText> errors) throws UserManagementException
    {
        super.additionalSignupOperations(createdUser, errors);
        
        Request request = _getRequest();
        
        String siteName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME);
        String language = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        
        if (zoneItem != null)
        {
            String userContentTypeId = zoneItem.getServiceParameters().getValue("user-content-type");
            if (StringUtils.isNotBlank(userContentTypeId))
            {
                ContentType userContentType = _contentTypeEP.getExtension(userContentTypeId);
                if (userContentType != null)
                {
                    // Create the user content type linked to this user
                    @SuppressWarnings("unchecked")
                    Map<String, Object> userValues = (Map<String, Object>) request.getAttribute(UserSignupManager.class.getName() + "$userValues");
                    
                    // Add link to created user
                    userValues.put("user", createdUser.getIdentity());
                        
                    if (errors.isEmpty())
                    {
                        try
                        {
                            _foHelper.createAndEditContent(_initWorkflowActionId, userContentType.getId(), siteName, createdUser.getFullName(), createdUser.getFullName(), language, userValues, _workflowName, null);
                        }
                        catch (AmetysRepositoryException | WorkflowException e)
                        {
                            getLogger().error("Unable to create the user content for user '{}' after sign up", createdUser, e);
                            errors.put("global", new I18nizableText("PLUGINS_USER_DIRECTORY_SIGNUP_SERVICE_CREATE_CONTENT_ERROR"));
                        }
                    }
                }
            }
        }
    }
    
}
